#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/string/builder.h>

#include <wmconsole/version3/wmcutil/log.h>

#include "stage_utils.h"

namespace NWebmaster {
namespace NStageUtils {

struct TTableTimeGreater {
    bool operator() (const NMR::TTableInfo &x, const NMR::TTableInfo &y) const {
        return x.Time > y.Time;
    }
};

void AppendRecord(NMR::TServer &server, const TString &table, const TString &key, const TString &subKey, const TString &value) {
    NMR::TClient client(server);
    NMR::TUpdate output(client, MrPath(table), NMR::UM_APPEND);
    output.AddSub(key, subKey, value);
}

void CreateDummyTable(NMR::TServer &server, const TString &dummyTable, bool overwrite) {
    NMR::TTableInfo info;
    if (!overwrite) {
        if (GetTableInfo(server, dummyTable, info)) {
            return;
        }
    }
    NMR::TClient client(server);
    NMR::TUpdate output(client, MrPath(dummyTable));
}

bool GetLatestTable(NMR::TServer &server, const TString &table_prefix, TString &tableName) {
    TVector<NMR::TTableInfo> allTablesInfo;

    try {
        server.GetTables(&allTablesInfo, NMR::GT_PREFIX_MATCH, MrPath(table_prefix).data());
    } catch (const std::exception &e) {
        return false;
    }

    if (allTablesInfo.empty()) {
        return false;
    }

    ui32 ts = 0;

    for (TVector<NMR::TTableInfo>::const_iterator it = allTablesInfo.begin(); it != allTablesInfo.end(); ++it) {
        if (it->Time > ts) {
            ts = it->Time;
            tableName = it->Name;
        }
    }

    return true;
}

void GetPathAlternative(NMR::TServer &server, const TString &sourcePrefix, TSet<TString> &alts, const char *pathDel) {
    TVector<NMR::TTableInfo> tables;
    server.GetTables(&tables, NMR::GT_PREFIX_MATCH, MrPath(sourcePrefix).data());

    for (const auto &_table : tables) {
        TString table(_table.Name);
        TString rest = table.substr(sourcePrefix.size());
        bool trailingSlash = (rest.find(pathDel) == 0);
        if (trailingSlash) {
            rest = rest.substr(1);
        }
        size_t del = rest.find(pathDel);
        TString ts = rest.substr(0, del);
        if (trailingSlash) {
            ts = pathDel + ts;
        }
        alts.insert(ts);
    }
}

bool GetTableInfo(NMR::TServer &server, const TString &tableName, NMR::TTableInfo &tableInfo) {
    TVector<NMR::TTableInfo> tables;

    server.GetTables(&tables, NMR::GT_EXACT_MATCH, MrPath(tableName).data());
    if (tables.empty()) {
        return false;
    }

    tableInfo = tables.front();
    return true;
}

TString GetTempTable(const TString &prefix, const TString &name) {
    return JoinPathMR(prefix, name + "-" + ToString(Now().MicroSeconds()));
}

size_t AppendTableList(NMR::TServer &server, const TString &tablePrefix, TVector<TString> &appendTo, size_t maxCount) {
    TVector<TString> tmpList;
    size_t result = GetTableList(server, tablePrefix, tmpList, maxCount);
    appendTo.insert(appendTo.end(), tmpList.begin(), tmpList.end());
    return result;
}

size_t GetTableList(NMR::TServer &server, const TString &tablePrefix, TVector<NMR::TTableInfo> &tableList, size_t maxCount) {
    server.GetTables(&tableList, NMR::GT_PREFIX_MATCH, MrPath(tablePrefix).data());

    std::sort(tableList.begin(), tableList.end(), TTableTimeGreater());

    if (tableList.size() > maxCount) {
        tableList.resize(maxCount);
    }

    return tableList.size();
}

size_t GetTableList(NMR::TServer &server, const TString &tablePrefix, TVector<TString> &tableList, size_t maxCount) {
    TVector<NMR::TTableInfo> rawTableList;
    GetTableList(server, tablePrefix, rawTableList, maxCount);

    tableList.clear();
    for (TVector<NMR::TTableInfo>::const_iterator it = rawTableList.begin(); it != rawTableList.end(); ++it) {
        tableList.push_back(it->Name);
    }

    return tableList.size();
}

size_t GetTableListByTime(NMR::TServer &server, const TString &tablePrefix, TVector<TString> &tableList, const TDuration &duration) {
    TVector<NMR::TTableInfo> rawTableList;
    GetTableList(server, tablePrefix, rawTableList, Max<size_t>());

    TInstant now = TInstant::Now();

    tableList.clear();
    for (TVector<NMR::TTableInfo>::const_iterator it = rawTableList.begin(); it != rawTableList.end(); ++it) {
        if ((now - TInstant::Seconds(it->Time)) < duration) {
            tableList.push_back(it->Name);
        }
    }

    return tableList.size();
}

TString GetTableName(const TString &path) {
    size_t delimiter = path.rfind('/');

    if (delimiter == TString::npos) {
        return path;
    }

    return path.substr(delimiter + 1);
}

size_t DropTables(NMR::TServer &server, const TVector<TString> &table_list) {
    for (size_t i = 0; i < table_list.size(); i++) {
        const TString &table = table_list[i];
        try {
            server.Drop(MrPath(table));
            LOG_INFO("dropped table %s", table.data());
        } catch (const std::exception &e) {
            LOG_WARN("unable to drop table %s: %s", table.data(), e.what());
        }
    }

    return table_list.size();
}

void LoadRecords(NMR::TServer &server, const TString &tableName, TVector<TRecord> &records, size_t limit) {
    NMR::TClient client(server);
    NMR::TTable table(client, MrPath(tableName));
    records.clear();
    for (NMR::TTableIterator it = table.Begin(); it.IsValid() && records.size() < limit; ++it) {
        TString key = it.GetKey().AsString();
        TString subkey = it.GetSubKey().AsString();
        TString value = it.GetValue().AsString();
        records.push_back(TRecord(key, subkey, value));
    }
}

bool IsEmpty(NMR::TServer &server, const TString &tableName) {
    return NMR::TClient(server).IsEmpty(MrPath(tableName));
}

bool LoadWebmastersHosts(NMR::TServer &server, const TString &tablePrefix, THashSet<TString> &hosts, TString &tableName, EHostScheme scheme) {
    if (!NStageUtils::GetLatestTable(server, tablePrefix, tableName)) {
        if (IsEmpty(server, tablePrefix)) {
            LOG_ERROR("there is no webmaster-hosts table with prefix %s", tablePrefix.data());
            return false;
        } else {
            tableName = tablePrefix;
        }
    }

    NMR::TClient client(server);
    NMR::TTable table(client, MrPath(tableName));
    for (NMR::TTableIterator it = table.Begin(); it.IsValid(); ++it) {
        TString host = it.GetKey().AsString();

        if (scheme == E_HOST_SCHEME_ROBOT) {
            if (host.find("https://") == TString::npos && host.find("ftp://") == TString::npos) {
                size_t del = host.find("://");
                if (del != TString::npos) {
                    host = host.substr(del + 3);
                }
            }
        } else if (scheme == E_HOST_SCHEME_NONE) {
            size_t del = host.find("://");
            if (del != TString::npos) {
                host = host.substr(del + 3);
            }
        }

        hosts.insert(host);
    }

    return true;
}

bool LoadWebmastersHosts(NMR::TServer &server, const TString &tablePrefix, THashSet<TString> &hosts, EHostScheme scheme) {
    TString destTable;
    return LoadWebmastersHosts(server, tablePrefix, hosts, destTable, scheme);
}

void LogTables(const TString &message, const TVector<TString> &table_list) {
    for (size_t i = 0; i < table_list.size(); i++) {
        const TString &table = table_list[i];
        LOG_INFO("%s %s", message.data(), table.data());
    }
}

void MoveTable(NMR::TServer &server, const TString &srcTable, const TString &dstTable) {
    NMR::TCopyParams params;
    params.AddInputTable(MrPath(srcTable));
    params.AddOutputTable(MrPath(dstTable));
    server.Move(params);
    server.Drop(MrPath(srcTable));
}

void SetSymbolicLinkYT(NYT::TProxy &proxy, TString from, TString to) {
    if (from.find("//") != 0) {
        from = "//" + from;
    }

    if (to.find("//") != 0) {
        to = "//" + to;
    }

    TString tx = proxy.StartTransaction();
    proxy.RemoveIfExists(to);
    proxy.Link(from, to);
    proxy.CommitTransaction(tx);
}

time_t GetModifyTimeYT(NYT::TProxy& proxy, const TString& path) {
    const TString attr = GetAttrYT(proxy, path, "modification_time");
    TInstant t = TInstant::ParseIso8601Deprecated(attr);
    return t.Seconds();
}

size_t GetRowCountYT(NYT::TProxy& proxy, const TString& path) {
    return FromString<size_t>(GetAttrYT(proxy, path, "row_count"));
}

TString GetSymbolicLinkYT(NYT::TProxy &proxy, const TString &table) {
    return GetAttrYT(proxy, table, "path");
}

TString GetAttrYT(NYT::TProxy &proxy, const TString& table, const TString& attr) {
    const TString path = TStringBuilder() << YtPath(table) << "/@" << attr;
    const TString value = proxy.Get(path);
    //Y_ENSURE(value[0] == '"' && value[+value - 1] == '"', "Unquoted return value `" << value << '`');
    return TString{TStringBuf(value).After('"').Before('"')};
}

void SetAttrYT(NYT::TProxy &proxy, const TString& table, const TString& attr, const TString& value) {
    const TString path = TStringBuilder() << YtPath(table) << "/@" << attr;
    const TString quotedValue = TStringBuilder() << "\"" << value << "\"";

    proxy.Set(path, quotedValue);
}

TString MrPath(const TString& table) {
    const TString ytPrefix = "//";

    if (table.StartsWith(ytPrefix)) {
        return table.substr(2);
    }

    return table;
}

TString YtPath(const TString &table) {
    const TString ytPrefix = "//";

    if (table.StartsWith(ytPrefix)) {
        return table;
    } else {
        if (table.StartsWith('/')) {
            return TStringBuilder() << "/" << table;
        } else {
            return TStringBuilder() << ytPrefix << table;
        }
    }
}

void SetArchiveAttrYT(NYT::TProxy& proxy, const TString& table, const TString &eCodec, const TString &cCodec) {
    const TString ytTable = YtPath(table);

    SetAttrYT(proxy, ytTable, "compression_codec", cCodec);
    SetAttrYT(proxy, ytTable, "erasure_codec", eCodec);
}

void ArchiveTableYT(NYT::TProxy &proxy, const TString& table, const TString &eCodec, const TString &cCodec) {
    SetArchiveAttrYT(proxy, table, eCodec, cCodec);

    const TString ytTable = YtPath(table);
    const TString spec = "{\"force_transform\":\"true\", \"combine_chunks\":\"true\"}";

    if (proxy.IsTableSorted(ytTable)) {
        proxy.SortedMerge(
            TVector<NMR::TInputTable>(1, ytTable),
            NMR::TUpdateTable(ytTable),
            spec
        );
    } else {
        proxy.UnorderedMerge(
            TVector<NMR::TInputTable>(1, ytTable),
            NMR::TUpdateTable(ytTable),
            spec
        );
    }
}

bool IsArchivedTableYT(NYT::TProxy &proxy, const TString &table) {
    const TString ytTable = YtPath(table);
    const TString compressionCodec = GetAttrYT(proxy, ytTable, "compression_codec");
    return GetAttrYT(proxy, ytTable, "erasure_codec").find("lrc") != TString::npos
        && (compressionCodec.find("gzip_best_compression") != TString::npos || compressionCodec.find("brotli") != TString::npos);
}

} //namespace NStageUtils
} //namespace NWebmaster
