#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <wmconsole/version3/wmcutil/http_client.h>

#include "transfer_manager.h"
#include "yt_utils.h"

namespace NWebmaster {

TTransferManager::TTaskInfo::TTaskInfo(const TString &jsonContent) {
    NJson::TJsonValue root;
    TStringStream json(jsonContent);
    NJson::ReadJsonTree(&json, &root);

    State = NameToState(root["state"].GetString());
    CreationTime = NYTUtils::GetIso8601Time(root["creation_time"].GetString());
    //StartTime = NYTUtils::GetIso8601Time(root["start_time"].GetString());
    DurationMinutes = (Now() - TInstant::Seconds(CreationTime)).Minutes();
}

TTransferManager::TTaskInfo::EState TTransferManager::TTaskInfo::NameToState(const TString &name) {
    if (name == "pending" || name == "running") {
        return RUNNING;
    } else if (name == "completed"  || name == "skipped") {
        return COMPLETED;
    } else if (name == "failed" || name == "aborted") {
        return FAILED;
    }

    ythrow yexception() << "TTransferManager: Parsing unknown state: " << name;
}

TTransferManager::TTransferManager(const TString &token, const TString &api, size_t timeout)
    : Api(api)
    , Timeout(timeout)
    , Token(token)
{
}

TString TTransferManager::GetCopyJsonParams(const TString &sourceCluster, const TString &sourceTable, const TString &destCluster, const TString &destTable) {
    TString jsonParams;
    TStringOutput jsonStream(jsonParams);

    NJson::TJsonWriter writer(&jsonStream, false, false);
    writer.OpenMap();
    writer.Write("source_table", sourceTable);
    writer.Write("source_cluster", TStringBuf(sourceCluster).Before('.'));
    writer.Write("destination_table", destTable);
    writer.Write("destination_cluster", TStringBuf(destCluster).Before('.'));
    writer.CloseMap();
    writer.Flush();

    return jsonParams;
}

bool TTransferManager::GetTaskInfo(const TString &taskId, TTaskInfo &info) try {
    const TString json = HttpGetRequest(Api + taskId + "/");
    info = TTaskInfo(json);
    return true;
} catch (const NNeh::TError &e) {
    if (e.Code == 404) {
        return false;
    }
    throw;
}

TString TTransferManager::HttpGetRequest(const TString &handler) {
    NNeh::TResponseRef r = NNeh::Request(handler)->Wait(TDuration::MilliSeconds(Timeout));
    if (!r) {
        ythrow yexception() << "unable to process HttpGet request";
    }

    if (r->IsError()) {
        throw NNeh::TError(r->GetErrorText(), r->GetErrorType(), r->GetErrorCode());
    }

    return TString(r->Data.data(), r->Data.size());
}

TString TTransferManager::HttpPostRequest(const TString &handler, const TString &data) {
    TVector<TString> HttpHeaders;
    HttpHeaders.push_back("Content-Type: application/json");
    HttpHeaders.push_back(TString("Authorization: OAuth ") + Token);
    NWebmaster::TPostRequest<TString> post(handler);
    post.SetTimeout(Timeout);
    post.SetHeaders(HttpHeaders);
    post.SetData(data);
    TAtomicSharedPtr<TString> res = post.Perform();
    return *res;
}

TString TTransferManager::PostTask(const TString &sourceCluster, const TString &sourceTable, const TString &destCluster, const TString &destTable) {
    const TString jsonParams = GetCopyJsonParams(sourceCluster, sourceTable, destCluster, destTable);
    const TString taskId = HttpPostRequest(Api, jsonParams);
    LogInfo(TStringBuilder() << "TTransferManager: posted task " << taskId);
    return taskId;
}

void TTransferManager::PostTaskAndWait(const TString &sourceCluster, const TString &sourceTable, const TString &destCluster, const TString &destTable) {
    const TString taskId = PostTask(sourceCluster, sourceTable, destCluster, destTable);
    WaitForTaskDone(taskId);
}

void TTransferManager::WaitForTaskDone(const TString &taskId) {
    const static int ERROR_RETRY_TIMEOUT_SECONDS = 5;
    const static int POLL_RETRY_TIMEOUT_MINUTES = 5;
    while(true) {
        try {
            TTaskInfo info;
            if (GetTaskInfo(taskId, info)) {
                if (info.State == TTaskInfo::COMPLETED) {
                    LogInfo(TStringBuilder() << "TTransferManager: task " << taskId << " completed");
                    return;
                } else if (info.State == TTaskInfo::FAILED) {
                    ythrow yexception() << "TTransferManager: task " << taskId << " failed or aborted";
                } else {
                    if (info.DurationMinutes > 0) {
                        LogInfo(TStringBuilder() << "TTransferManager: task " << taskId << " is still running for " << info.DurationMinutes << " minutes");
                    }
                }
            } else {
                ythrow yexception() << "TTransferManager: task " << taskId << " does not exist";
            }
        } catch (yexception &e) {
            LogInfo(TStringBuilder() << "TTransferManager: error " << e.what() << ", retry in " << ERROR_RETRY_TIMEOUT_SECONDS << " seconds");
            Sleep(TDuration::Seconds(ERROR_RETRY_TIMEOUT_SECONDS));
            continue;
        }
        Sleep(TDuration::Minutes(POLL_RETRY_TIMEOUT_MINUTES));
    }
}

void TTransferManager::DefaultLogInfo(const TString &msg) {
    Cout << msg << Endl;
}

void (*TTransferManager::LogInfo)(const TString &msg) = &TTransferManager::DefaultLogInfo;
const char *TTransferManager::BANACH = "banach";
const char *TTransferManager::HAHN = "hahn";

} //namespace NWebmaster
