#pragma once

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/operation.h>
#include <util/generic/list.h>
#include <util/generic/map.h>
#include <util/thread/factory.h>
#include <util/string/printf.h>

#include "yt_utils.h"

#define MEMORY_LIMIT_1GB    1024ul * 1024ul * 1024ul
#define MEMORY_LIMIT_2GB    MEMORY_LIMIT_1GB * 2ul
#define MEMORY_LIMIT_3GB    MEMORY_LIMIT_1GB * 3ul
#define MEMORY_LIMIT_4GB    MEMORY_LIMIT_1GB * 4ul
#define MEMORY_LIMIT_5GB    MEMORY_LIMIT_1GB * 5ul
#define MEMORY_LIMIT_6GB    MEMORY_LIMIT_1GB * 6ul
#define MEMORY_LIMIT_7GB    MEMORY_LIMIT_1GB * 7ul
#define MEMORY_LIMIT_8GB    MEMORY_LIMIT_1GB * 8ul
#define MEMORY_LIMIT_9GB    MEMORY_LIMIT_1GB * 9ul
#define MEMORY_LIMIT_10GB   MEMORY_LIMIT_1GB * 10ul
#define MEMORY_LIMIT_11GB   MEMORY_LIMIT_1GB * 11ul
#define MEMORY_LIMIT_12GB   MEMORY_LIMIT_1GB * 12ul
#define MEMORY_LIMIT_13GB   MEMORY_LIMIT_1GB * 13ul
#define MEMORY_LIMIT_14GB   MEMORY_LIMIT_1GB * 14ul
#define MEMORY_LIMIT_15GB   MEMORY_LIMIT_1GB * 15ul
#define MEMORY_LIMIT_16GB   MEMORY_LIMIT_1GB * 16ul
#define MEMORY_LIMIT_17GB   MEMORY_LIMIT_1GB * 17ul
#define MEMORY_LIMIT_18GB   MEMORY_LIMIT_1GB * 18ul
#define MEMORY_LIMIT_19GB   MEMORY_LIMIT_1GB * 19ul
#define MEMORY_LIMIT_20GB   MEMORY_LIMIT_1GB * 20ul

#define SYNC_CTX   -1
#define ASYNC_CTX0  0
#define ASYNC_CTX1  1
#define ASYNC_CTX2  2
#define ASYNC_CTX3  3
#define ASYNC_CTX4  4
#define ASYNC_CTX5  5
#define ASYNC_CTX6  6
#define ASYNC_CTX7  7

namespace NWebmaster {
namespace NYTRunner {

typedef TAtomicSharedPtr<IThreadFactory::IThreadAble> TThreadAblePtr;

template <class TMapper>
struct TMapTask;

template <class TMapper, class TReduceCombiner, class Reducer>
struct TMapReduceTask;

struct TMergeTask;

template <class Reducer>
struct TReduceTask;

struct TSortTask;

struct TOpRunner {
    enum EReduceMode {
        REDUCE_NORMAL_MODE,
        REDUCE_JOIN_MODE,
    };

    struct TTable {
        enum EType {
            UNDEFINED,
            NODE,
            YAMR,
            PROTO,
        };

        TTable()
            : Type(NODE)
        {
        }

        TTable(const NYT::TRichYPath &tableName, EType type)
            : RichPath(tableName)
            , Type(type)
        {
        }

    public:
        NYT::TRichYPath RichPath;
        EType Type;
    };

    struct TParams {
        TParams()
            : GlobalSpec(NYT::TNode::CreateMap())
        {
        }

        void AddInputTable(const NYT::TRichYPath &input, TTable::EType type = TTable::NODE) {
            SrcTables.push_back(TTable(input, type));
        }

        void AddOutputTable(const NYT::TRichYPath &output, TTable::EType type = TTable::NODE) {
            DstTables.push_back(TTable(output, type));
        }

        template <class TOperationSpec>
        void CopyToOperationSpec(TOperationSpec &opSpec) {
            using namespace NYTUtils;

            for (const TTable &info : SrcTables) {
                const NYT::TRichYPath &path = info.RichPath;

                switch (info.Type) {
                case TTable::NODE:
                    opSpec.template AddInput<NYT::TNode>(path);
                    break;
                case TTable::YAMR:
                    opSpec.template AddInput<NYT::TYaMRRow>(path);
                    break;
                case TTable::PROTO:
                    opSpec.AddProtobufInput_VerySlow_Deprecated(path);
                    break;
                default:
                    ythrow yexception() << info.RichPath.Path_ << " undefined input table type";
                }
            }

            for (const TTable &info : DstTables) {
                const NYT::TRichYPath &path = info.RichPath;

                switch (info.Type) {
                case TTable::NODE:
                    opSpec.template AddOutput<NYT::TNode>(path);
                    break;
                case TTable::YAMR:
                    opSpec.template AddOutput<NYT::TYaMRRow>(path);
                    break;
                case TTable::PROTO:
                    opSpec.AddProtobufOutput_VerySlow_Deprecated(path);
                    break;
                default:
                    ythrow yexception() << info.RichPath.Path_ << " undefined output table type";
                }
            }
        }

        void UpdateOptions() {
            GlobalSpec["enable_legacy_live_preview"] = EnableLivePreviewFlag;
            Options.Spec(GlobalSpec);
        }

    public:
        TList<TTable> SrcTables;
        TList<TTable> DstTables;
        NYT::TUserJobSpec MapperUserJobSpec;
        NYT::TUserJobSpec CombinerUserJobSpec;
        NYT::TUserJobSpec ReducerUserJobSpec;
        NYT::TOperationOptions Options;
        NYT::TNode GlobalSpec;
        THolder<NYT::TSortColumns> JoinBy;
        THolder<NYT::TSortColumns> MergeBy;
        THolder<NYT::TSortColumns> ReduceBy;
        THolder<NYT::TSortColumns> SortBy;
        size_t JobCount = 0;
        size_t DataSizePerJob = 0;
        size_t PartitionCount = 0;
        size_t PartitionDataSize = 0;
        bool EnableLivePreviewFlag = false;

        bool OrderedMap = false;
    };

    using TParamsPtr = TSimpleSharedPtr<TParams>;

    TOpRunner(NYT::IClientBasePtr client);

private:
    TString GetCommentedMsg(const TString &msg) const;
    void CommentLogInfo(TString msg) const;

    TOpRunner& InputImpl(const NYT::TRichYPath &input, TTable::EType type);
    TOpRunner& InputByPrefixImpl(const NYT::TRichYPath &prefix, TTable::EType type);
    TOpRunner& OutputImpl(const NYT::TRichYPath &output, TTable::EType type);

public:
    TOpRunner& Archive(const TString& table, int ctx = SYNC_CTX, size_t desiredChunkCount = 500, const TString& eCodec = "lrc_12_2_2", const TString& cCodec = "brotli8");
    TOpRunner& Defrag(const TString& table, int ctx = SYNC_CTX, size_t desiredChunkCount = 500);

    TOpRunner& Input(const NYT::TRichYPath &input);
    TOpRunner& InputNode(const NYT::TRichYPath &input);
    TOpRunner& InputProto(const NYT::TRichYPath &input);
    TOpRunner& InputYaMR(const NYT::TRichYPath &input);

    TOpRunner& InputByPrefix(const NYT::TRichYPath &prefix);
    TOpRunner& InputNodeByPrefix(const NYT::TRichYPath &prefix);
    TOpRunner& InputProtoByPrefix(const NYT::TRichYPath &prefix);
    TOpRunner& InputYaMRByPrefix(const NYT::TRichYPath &prefix);

    TOpRunner& Output(const NYT::TRichYPath &input);
    TOpRunner& OutputNode(const NYT::TRichYPath &input);
    TOpRunner& OutputProto(const NYT::TRichYPath &output);
    TOpRunner& OutputYaMR(const NYT::TRichYPath &output);

    TOpRunner& Comment(const TString &comment);
    TOpRunner& Copy(const TString &src, const TString &dst);
    TOpRunner& Drop(const TString &table);
    TOpRunner& DropByPrefix(const TString &prefix);
    //TOpRunner& EnableReduceWithoutSort();
    TOpRunner& EnableLivePreview();
    TOpRunner& EnableOrderedMap();

    TOpRunner& JoinBy(const NYT::TSortColumns &keys);

    template <class... TArgs>
    TOpRunner& JoinBy(const TArgs&... args) {
        return JoinBy(NYT::TSortColumns(args...));
    }

    template <class TReducer>
    TOpRunner& JoinReduce(TReducer reducer, int ctx = SYNC_CTX) {
        PrepareParams();

        if (!Params->JoinBy) {
            ythrow yexception() << "join reduce: JoinBy columns are not set";
        }

        TString opComment = GetCommentedMsg(TString("join reduce operation ") + GetParametersComment(*Params));
        return RunTask(new TReduceTask<TReducer>(Client, reducer, Params, REDUCE_JOIN_MODE, opComment), ctx);
    }

    template <class TMapper>
    TOpRunner& Map(TMapper mapper, int ctx = SYNC_CTX) {
        PrepareParams();
        TString opComment = GetCommentedMsg(TString("map operation ") + GetParametersComment(*Params));
        return RunTask(new TMapTask<TMapper>(Client, mapper, Params, opComment), ctx);
    }

    template <class TReducer>
    TOpRunner& MapReduce(TReducer reducer, int ctx = SYNC_CTX) {
        return MapReduce<nullptr_t, nullptr_t, TReducer>(nullptr, nullptr, reducer, ctx);
    }

    template <class TMapper, class TReducer>
    TOpRunner& MapReduce(TMapper mapper, TReducer reducer, int ctx = SYNC_CTX) {
        return MapReduce<TMapper, nullptr_t, TReducer>(mapper, nullptr, reducer, ctx);
    }

    template <class TMapper, class TReduceCombiner, class TReducer>
    TOpRunner& MapReduce(TMapper mapper, TReduceCombiner combiner, TReducer reducer, int ctx = SYNC_CTX) {
        PrepareParams();

        if (!Params->ReduceBy) {
            ythrow yexception() << "mapreduce: ReduceBy columns are not set";
        }

        TString opComment = GetCommentedMsg(TString("mapreduce operation ") + GetParametersComment(*Params));
        return RunTask(new TMapReduceTask<TMapper, TReduceCombiner, TReducer>(Client, mapper, combiner, reducer, Params, opComment), ctx);
    }

    TOpRunner& Merge(int ctx = SYNC_CTX);
    TOpRunner& MergeBy(const NYT::TSortColumns &keys);

    template <class... TArgs>
    TOpRunner& MergeBy(const TArgs&... args) {
        return MergeBy(NYT::TSortColumns(args...));
    }

    TOpRunner& Move(const TString &src, const TString &dst);

    template <class TReducer>
    TOpRunner& Reduce(TReducer reducer, int ctx = SYNC_CTX) {
        PrepareParams();

        if (!Params->ReduceBy) {
            ythrow yexception() << "reduce: ReduceBy columns are not set";
        }

        TString opComment = GetCommentedMsg(TString("reduce operation ") + GetParametersComment(*Params));
        return RunTask(new TReduceTask<TReducer>(Client, reducer, Params, REDUCE_NORMAL_MODE, opComment), ctx);
    }

    TOpRunner& ReduceBy(const NYT::TSortColumns &keys);

    template <class... TArgs>
    TOpRunner& ReduceBy(const TArgs&... args) {
        return ReduceBy(NYT::TSortColumns(args...));
    }

    TOpRunner& Sort(const TString &src, int ctx = SYNC_CTX);
    TOpRunner& Sort(const TString &src, const TString &dst, int ctx = SYNC_CTX);

    TOpRunner& SortBy(const NYT::TSortColumns &keys);

    template <class... TArgs>
    TOpRunner& SortBy(const TArgs&... args) {
        return SortBy(NYT::TSortColumns(args...));
    }

    template <class T>
    TOpRunner& Spec(const TString &name, const T &value) {
        PrepareParams();
        Params->GlobalSpec[name] = value;
        return *this;
    }

    template <class T>
    TOpRunner& Title(const T &title) {
        PrepareParams();
        Params->GlobalSpec["title"] = title;
        return *this;
    }

    template <class T>
    TOpRunner& MaxDataSizePerJob(const T &maxDataSizePerJob) {
        PrepareParams();
        Params->GlobalSpec["max_data_size_per_job"] = maxDataSizePerJob;
        return *this;
    }

    TOpRunner& File(const TString &file);
    TOpRunner& LocalFile(const TString &file);
    TOpRunner& MaxRowWeight(size_t bytes);

    TOpRunner& MemoryLimit(size_t n);
    TOpRunner& MapperMemoryLimit(size_t limit);
    TOpRunner& CombinerMemoryLimit(size_t limit);
    TOpRunner& ReducerMemoryLimit(size_t limit);

    TOpRunner& DataSizePerJob(size_t dspj);
    TOpRunner& JobCount(size_t jc);
    TOpRunner& PartitionCount(size_t pc);
    TOpRunner& PartitionDataSize(size_t pds);

    //to avoid "Detected excessive disk IO"
    TOpRunner& UseTmpfsInMapper();
    TOpRunner& UseTmpfsInReduceCombiner();
    TOpRunner& UseTmpfsInReducer();
    TOpRunner& UseTmpfs();

    TOpRunner& Wait(size_t ctx);

public:
    void PrepareParams() {
        if (!Params) {
            Params.Reset(new TParams);
        }
    }

    template<typename T>
    TOpRunner& RunTask(T *task, int ctx) {
        try {
            TThreadAblePtr ptr(task);

            if (ctx == SYNC_CTX) {
                task->DoExecute();
            } else {
                task->RunAsync();
                CtxQueue[ctx].push_back(ptr);
            }
        } catch(...) {
            Params.Reset();
            throw;
        }
        Params.Reset();
        return *this;
    }

    static TString GetParametersComment(TParams &params);
    static void DefaultLogInfo(const TString &msg);

public:
    TMap<size_t, TList<TThreadAblePtr>> CtxQueue;
    TString CommentStr;
    NYT::IClientBasePtr Client;
    TParamsPtr Params;

public:
    static void (*LogInfo)(const TString &msg);
};

template <class TMapper>
struct TMapTask : public IThreadFactory::IThreadAble {
    TMapTask(NYT::IClientBasePtr client, TMapper mapOperator, TOpRunner::TParamsPtr params, const TString &comment = "")
        : Client(client)
        , MapOperator(mapOperator)
        , Params(params)
        , Comment(comment)
    {
    }

    ~TMapTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:
    void DoExecute() override {
        TOpRunner::LogInfo(Comment);

        NYT::TMapOperationSpec opSpec;

        if (Params->OrderedMap) {
            opSpec.Ordered(true);
        }

        if (Params->JobCount != 0) {
            opSpec.JobCount(Params->JobCount);
        }

        if (Params->DataSizePerJob != 0) {
            opSpec.DataSizePerJob(Params->DataSizePerJob);
        }

        Params->CopyToOperationSpec(opSpec);
        Params->UpdateOptions();
        opSpec.MapperSpec(Params->MapperUserJobSpec);
        Client->Map(opSpec, MapOperator, Params->Options);

        TOpRunner::LogInfo(Comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    TMapper MapOperator;
    TOpRunner::TParamsPtr Params;
    TString Comment;
};

template<class TReducer>
static void RunMapReduce(NYT::IClientBasePtr client, TOpRunner::TParamsPtr params, const NYT::TMapReduceOperationSpec &opSpec, nullptr_t, nullptr_t, TReducer r) {
    client->MapReduce(opSpec, nullptr, r, params->Options);
}

template<class TMapper, class TReducer>
static void RunMapReduce(NYT::IClientBasePtr client, TOpRunner::TParamsPtr params, const NYT::TMapReduceOperationSpec &opSpec, TMapper m, nullptr_t, TReducer r) {
    client->MapReduce(opSpec, m, r, params->Options);
}

template <class TMapper, class TReduceCombiner, class TReducer>
static void RunMapReduce(NYT::IClientBasePtr client, TOpRunner::TParamsPtr params, const NYT::TMapReduceOperationSpec &opSpec, TMapper m, TReduceCombiner c, TReducer r) {
    client->MapReduce(opSpec, m, c, r, params->Options);
}

template <class TMapper, class TReduceCombiner, class TReducer>
struct TMapReduceTask : public IThreadFactory::IThreadAble {
    TMapReduceTask(NYT::IClientBasePtr client, TMapper mapOperator, TReduceCombiner combineOperator, TReducer reduceOperator, TOpRunner::TParamsPtr params, const TString &comment = "")
        : Client(client)
        , MapOperator(mapOperator)
        , CombineOperator(combineOperator)
        , ReduceOperator(reduceOperator)
        , Params(params)
        , Comment(comment)
    {
    }

    ~TMapReduceTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:
    void DoExecute() override {
        TOpRunner::LogInfo(Comment);

        NYT::TMapReduceOperationSpec opSpec;
        Params->CopyToOperationSpec(opSpec);
        Params->UpdateOptions();
        opSpec.ReduceBy(*Params->ReduceBy);

        if (Params->SortBy) {
            opSpec.SortBy(*Params->SortBy);
        }

        if (Params->JobCount != 0) {
            opSpec.MapJobCount(Params->JobCount);
        }

        if (Params->DataSizePerJob != 0) {
            opSpec.DataSizePerMapJob(Params->DataSizePerJob);
        }

        if (Params->PartitionCount != 0) {
            opSpec.PartitionCount(Params->PartitionCount);
        }

        if (Params->PartitionDataSize != 0) {
            opSpec.PartitionDataSize(Params->PartitionDataSize);
        }

        opSpec.MapperSpec(Params->MapperUserJobSpec);
        opSpec.ReduceCombinerSpec(Params->CombinerUserJobSpec);
        opSpec.ReducerSpec(Params->ReducerUserJobSpec);

        RunMapReduce(Client, Params, opSpec, MapOperator, CombineOperator, ReduceOperator);

        TOpRunner::LogInfo(Comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    TMapper MapOperator;
    TReduceCombiner CombineOperator;
    TReducer ReduceOperator;
    TOpRunner::TParamsPtr Params;
    TString Comment;
};

template <class TReducer>
struct TReduceTask : public IThreadFactory::IThreadAble {
    TReduceTask(NYT::IClientBasePtr client, TReducer reduceOperator, TOpRunner::TParamsPtr params, TOpRunner::EReduceMode eReduceMode, const TString &comment = "")
        : Client(client)
        , ReduceOperator(reduceOperator)
        , Params(params)
        , Mode(eReduceMode)
        , Comment(comment)
    {
    }

    ~TReduceTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:

    void DoExecuteNormal() {
        NYT::TReduceOperationSpec opSpec;
        Params->CopyToOperationSpec(opSpec);

        if (Params->SortBy) {
            opSpec.SortBy(*Params->SortBy);
        }

        if (Params->JoinBy) {
            opSpec.JoinBy(*Params->JoinBy);
        }

        if (Params->JobCount != 0) {
            opSpec.JobCount(Params->JobCount);
        }

        if (Params->DataSizePerJob != 0) {
            opSpec.DataSizePerJob(Params->DataSizePerJob);
        }

        opSpec.ReducerSpec(Params->ReducerUserJobSpec);
        opSpec.ReduceBy(*Params->ReduceBy);
        Client->Reduce(opSpec, ReduceOperator, Params->Options);
    }

    void DoExecuteJoin() {
        NYT::TJoinReduceOperationSpec opSpec;
        Params->CopyToOperationSpec(opSpec);
        opSpec.ReducerSpec(Params->ReducerUserJobSpec);
        opSpec.JoinBy(*Params->JoinBy);
        Client->JoinReduce(opSpec, ReduceOperator, Params->Options);
    }

    void DoExecute() override {
        TOpRunner::LogInfo(Comment);
        Params->UpdateOptions();
        if (Mode == TOpRunner::REDUCE_NORMAL_MODE) {
            DoExecuteNormal();
        } else if (Mode == TOpRunner::REDUCE_JOIN_MODE) {
            DoExecuteJoin();
        } else {
            ythrow yexception() << "unknown reduce mode";
        }
        TOpRunner::LogInfo(Comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    TReducer ReduceOperator;
    TOpRunner::TParamsPtr Params;
    TOpRunner::EReduceMode Mode;
    TString Comment;
};

struct TSortTask : public IThreadFactory::IThreadAble {
    TSortTask(NYT::IClientBasePtr client, TOpRunner::TParamsPtr params, const TString &src, const TString &dst = "")
        : Client(client)
        , Params(params)
        , Src(src)
        , Dst(dst.empty() ? src : dst)
    {
    }

    ~TSortTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:
    void DoExecute() override {
        TString comment = Sprintf("sort operation [input: %s; output: %s]", Src.data(), Dst.data());
        TOpRunner::LogInfo(comment);

        Params->UpdateOptions();

        Client->Sort(
            NYT::TSortOperationSpec()
                .AddInput(Src)
                .Output(NYT::TRichYPath(Dst).Append(false))
                .SortBy(*Params->SortBy),
            Params->Options
        );

        TOpRunner::LogInfo(comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    TOpRunner::TParamsPtr Params;
    TString Src;
    TString Dst;
};

struct TArchiveDefragTask : public IThreadFactory::IThreadAble {
    TArchiveDefragTask(NYT::IClientBasePtr client, const TString &table, size_t desiredChunkCount, const TString &eCodec, const TString &cCodec)
        : Client(client)
        , Table(table)
        , ChunkCount(desiredChunkCount)
        , ErasureCodec(eCodec)
        , CompressionCodec(cCodec)
        , DefragOnly(false)
    {
    }

    TArchiveDefragTask(NYT::IClientBasePtr client, const TString &table, size_t desiredChunkCount)
        : Client(client)
        , Table(table)
        , ChunkCount(desiredChunkCount)
        , DefragOnly(true)
    {
    }

    ~TArchiveDefragTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:
    void DoExecute() override {
        TString comment;
        if (DefragOnly) {
            comment = Sprintf("defrag operation [table: %s; chunks: %lu]", Table.data(), ChunkCount);
        } else {
            comment = Sprintf("archive operation [table: %s; chunks: %lu; erasure_codec: %s; compression_codec: %s]", Table.data(), ChunkCount, ErasureCodec.data(), CompressionCodec.data());
        }

        TOpRunner::LogInfo(comment);
        if (DefragOnly) {
            NYTUtils::DefragTable(Client, Table, ChunkCount);
        } else {
            NYTUtils::ArchiveTable(Client, Table, ChunkCount, ErasureCodec, CompressionCodec);
        }
        TOpRunner::LogInfo(comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    const TString Table;
    const size_t ChunkCount;
    const TString ErasureCodec;
    const TString CompressionCodec;
    const bool DefragOnly;
};

struct TMergeTask : public IThreadFactory::IThreadAble {
    TMergeTask(NYT::IClientBasePtr client, TOpRunner::TParamsPtr params, const TString &comment = "")
        : Client(client)
        , Params(params)
        , Comment(comment)
    {
    }

    ~TMergeTask() override {
        if (ThreadRef) {
            ThreadRef->Join();
        }
    }

public:
    void DoExecute() override {
        TOpRunner::LogInfo(Comment);

        NYT::TMergeOperationSpec opSpec;
        Params->UpdateOptions();

        NYT::EMergeMode mode = NYT::MM_SORTED;

        for (const TOpRunner::TTable &info : Params->SrcTables) {
            opSpec.AddInput(info.RichPath);

            if (!NYTUtils::IsTableSorted(Client, info.RichPath.Path_)) {
                mode = NYT::MM_UNORDERED;
            }
        }

        if (Params->DstTables.size() != 1) {
            ythrow yexception() << "merge incorrect output table count";
        }

        NYT::TRichYPath output(Params->DstTables.begin()->RichPath);
        output.Append(false);
        opSpec.Mode(mode);
        opSpec.Output(output);

        if (Params->MergeBy) {
            opSpec.MergeBy(*Params->MergeBy);
        }

        if (Params->JobCount != 0) {
            opSpec.JobCount(Params->JobCount);
        }

        if (Params->DataSizePerJob != 0) {
            opSpec.DataSizePerJob(Params->DataSizePerJob);
        }

        Client->Merge(opSpec, Params->Options);
        TOpRunner::LogInfo(Comment + " - done");
    }

    void RunAsync() {
        ThreadRef = SystemThreadFactory()->Run(this);
    }

public:
    TAutoPtr<IThreadFactory::IThread> ThreadRef;
    NYT::IClientBasePtr Client;
    TOpRunner::TParamsPtr Params;
    TString Comment;
};

} //namespace NYtRunner

using TOpRunner = NYTRunner::TOpRunner;

} //namespace NWebmaster
