#include <util/string/cast.h>

#include <mapreduce/yt/interface/logging/logger.h>
#include <mapreduce/yt/interface/errors.h>
#include <mapreduce/yt/interface/serialize.h>

#include "yt_utils.h"

namespace NWebmaster {
namespace NYTUtils {

void ArchiveTable(NYT::IClientBasePtr client, const TString& table, size_t desiredChunkCount, const TString& eCodec, const TString& cCodec) {
    SetArchiveAttr(client, table, eCodec, cCodec);
    DefragTable(client, table, desiredChunkCount);
}

void DefragTable(NYT::IClientBasePtr client, const TString& table, size_t desiredChunkCount) {
    NYT::EMergeMode mode = NYT::MM_UNORDERED;

    if (IsTableSorted(client, table)) {
        mode = NYT::MM_SORTED;
    }

    size_t uncompressedDataSize = GetAttr(client, table, "uncompressed_data_size").AsInt64();
    size_t dataSizePerJob = uncompressedDataSize / desiredChunkCount;

    client->Merge(
        NYT::TMergeOperationSpec()
            .Mode(mode)
            .CombineChunks(true)
            .ForceTransform(true)
            .AddInput(table)
            .Output(table),
        NYT::TOperationOptions().Spec(
            NYT::TNode()
                ("data_size_per_job", dataSizePerJob)
                ("job_count", desiredChunkCount)
        )
    );
}

void CreateDummyTable(NYT::IClientBasePtr client, const TString &dummyTable, bool overwrite) {
    if (!overwrite) {
        if (client->Exists(dummyTable)) {
            return;
        }
    }

    client->Remove(dummyTable,
        NYT::TRemoveOptions()
            .Force(true)
            .Recursive(true)
    );

    client->Create(dummyTable,
        NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
    );
}

void CreatePath(NYT::IClientBasePtr client, const TString &path) {
    if (!client->Exists(path)) {
        client->Create(path,
            NYT::NT_MAP,
            NYT::TCreateOptions()
                .Recursive(true)
        );
    }
}

void CreateTable(NYT::IClientBasePtr client, const TString &path, bool overwrite) {
    client->Create(path,
                   NYT::NT_TABLE,
                   NYT::TCreateOptions()
                           .Recursive(true)
                           .Force(overwrite)
    );
}

void DownloadFile(NYT::IClientBasePtr client, const TString& srcPath, const TString& dstPath) {
    TFixedBufferFileOutput writer(dstPath);
    NYT::IFileReaderPtr reader = client->CreateFileReader(srcPath);
    TransferData(reader.Get(), &writer);
    writer.Finish();
}

NYT::TTableSchema DropSortOrder(const NYT::TTableSchema &schema) {
    NYT::TTableSchema newSchema;
    newSchema.Strict(schema.Strict());
    for (const NYT::TColumnSchema &column : schema.Columns()) {
        newSchema.AddColumn(NYT::TColumnSchema().Name(column.Name()).Type(column.Type()));
    }
    return newSchema;
}

NYT::TNode GetAttr(NYT::IClientBasePtr client, const TString& table, const TString& attr) {
    const TString path = TStringBuilder() << table << "/@" << attr;
    return client->Get(path);
}

TString GetDirectoryName(const TString &path) {
    return TString{TStringBuf(path).RBefore('/')};
}

time_t GetIso8601Time(const TString &t) {
    return TInstant::ParseIso8601Deprecated(t).Seconds();
}

bool GetLatestTable(NYT::IClientBasePtr client, const TString &tablePrefix, TString &tableName) {
    TDeque<TTableInfo> allTablesInfo;
    GetTableList(client, tablePrefix, allTablesInfo, Max<size_t>());

    if (allTablesInfo.empty()) {
        return false;
    }

    time_t ts = 0;
    for (const TTableInfo &table : allTablesInfo) {
        if (table.Time > ts) {
            ts = table.Time;
            tableName = table.Name;
        }
    }

    return true;
}

time_t GetModificationTime(NYT::IClientBasePtr client, const TString &path) {
    const TString content = GetAttr(client, path, "modification_time").AsString();
    return GetIso8601Time(content);
}

bool GetTableInfo(NYT::IClientBasePtr client, const TString &tableName, TTableInfo &tableInfo) {
    TDeque<NYTUtils::TTableInfo> tables;
    GetTableList(client, tableName, tables);

    if (tables.size() == 1) {
        tableInfo = tables[0];
        return true;
    }

    return false;
}

size_t GetTableList(NYT::IClientBasePtr client, const TString &tablePrefix, TDeque<TTableInfo> &tableList, size_t maxCount) {
    TDeque<NYTUtils::TResolvedNode> dest;

    NYTUtils::ResolveTree(client, tablePrefix, dest,
        "modification_time",
        "row_count",
        "chunk_count",
        "resource_usage", //disk_space
        "compressed_data_size",
        "uncompressed_data_size",
        "replication_factor"
    );

    tableList.clear();
    for (const NYTUtils::TResolvedNode &node : dest) {
        if (node.Value.GetType() == NYT::TNode::Null && node.GetAttr("/type").AsString() == "table") {
            TTableInfo info(node.Path);
            info.Time = GetIso8601Time(node.GetAttr("/modification_time").AsString());
            info.RecordCount = node.GetAttr("/row_count").AsInt64();
            info.ChunkCount = node.GetAttr("/chunk_count").AsInt64();
            info.DiskSize = node.GetAttr("/resource_usage/disk_space").AsInt64();
            info.CompressedSize = node.GetAttr("/compressed_data_size").AsInt64();
            info.UncompressedSize = node.GetAttr("/uncompressed_data_size").AsInt64();
            info.ReplicationFactor = node.GetAttr("/replication_factor").AsInt64();
            tableList.push_back(info);
        }
    }

    std::sort(tableList.begin(), tableList.end(), NYTUtils::TTableInfo::TTimeGreater());

    if (tableList.size() > maxCount) {
        tableList.resize(maxCount);
    }

    return tableList.size();
}

size_t GetTableListByPrefix(NYT::IClientBasePtr client, const TString &prefix, TDeque<NYTUtils::TTableInfo> &tablesDst, size_t maxCount) {
    const TString root = NYTUtils::GetDirectoryName(prefix);
    TDeque<NYTUtils::TTableInfo> tables;
    TDeque<NYTUtils::TTableInfo> tmp;
    NYTUtils::GetTableList(client, root, tables, maxCount);
    for (const NYTUtils::TTableInfo &table : tables) {
        if (table.Name.StartsWith(prefix)) {
            tmp.push_back(table);
        }
    }
    tmp.swap(tablesDst);
    return tablesDst.size();
}

TString GetObjectName(const TString &path) {
    return TString{TStringBuf(path).RAfter('/')};
}

TString GetTableName(const TString &path) {
    return GetObjectName(path);
}

NYT::TTableSchema GetTableSchema(NYT::IClientBasePtr client, const NYT::TRichYPath &path) {
    NYT::TNode schemaNode = NYTUtils::GetAttr(client, path.Path_, "schema");
    NYT::TTableSchema tableSchema;
    NYT::Deserialize(tableSchema, schemaNode);
    return tableSchema;
}

TString GetTempTable(const TString &prefix, const TString &name) {
    return JoinPath(prefix, name + "-" + ToString(Now().MicroSeconds()));
}

NYT::TTableSchema GetYaMRSchema() {
    const static char *F_KEY       = "key";
    const static char *F_SUBKEY    = "subkey";
    const static char *F_VALUE     = "value";
    NYT::TTableSchema YaMRSchema;
    YaMRSchema.Strict(true);
    YaMRSchema.AddColumn(NYT::TColumnSchema().Name(F_KEY).Type(NYT::VT_STRING));
    YaMRSchema.AddColumn(NYT::TColumnSchema().Name(F_SUBKEY).Type(NYT::VT_STRING));
    YaMRSchema.AddColumn(NYT::TColumnSchema().Name(F_VALUE).Type(NYT::VT_STRING));
    return YaMRSchema;
}

void DisableLogger() {
    NYT::SetLogger(nullptr);
}

bool HasAttr(NYT::IClientBasePtr client, const TString &table, const TString &attr) {
    const TString path = TStringBuilder() << table << "/@" << attr;
    return client->Exists(path);
}

bool IsTableArchived(NYT::IClientBasePtr client, const TString &table) {
    const TString compressionCodec = GetAttr(client, table, "compression_codec").AsString();
    return GetAttr(client, table, "erasure_codec").AsString().Contains("lrc")
        && (compressionCodec.Contains("gzip") || compressionCodec.Contains("brotli") || compressionCodec.Contains("zlib"));
}

bool IsTableSorted(NYT::IClientBasePtr client, const TString& path) {
    return GetAttr(client, path, "sorted").AsBool();
}

static bool IsTrue(const NYT::TNode &node) {
    switch (node.GetType()) {
    case NYT::TNode::String:
        return node.AsString() == "true";
    case NYT::TNode::Bool:
        return node.AsBool();
    default:
        ythrow yexception() << "unable to infer boolean value";
    }
}

void PatchSchema(NYT::TTableSchema &tableSchema, const NYT::TColumnSchema &newColumn) {
    for (NYT::TColumnSchema &column : tableSchema.MutableColumns()) {
        if (column.Name() == newColumn.Name()) {
            column = newColumn;
            return;
        }
    }
    tableSchema.AddColumn(newColumn);
}

void PatchSchema(NYT::TTableSchema &tableSchema, const TString columnName, NYT::EValueType valueType) {
    PatchSchema(tableSchema, NYT::TColumnSchema().Name(columnName).Type(valueType));
}

void RemoveAttr(NYT::IClientBasePtr client, const TString& table, const TString& attr) {
    const TString path = TStringBuilder() << table << "/@" << attr;
    if (client->Exists(path)) {
        client->Remove(path);
    }
}

void ResolveTreeAttr(const TString &path, const NYT::TNode &root, TDeque<TResolvedNode> &dest) {
    for (const auto &obj : root.AsMap()) {
        const TString newPath = path + "/" + obj.first;
        if (obj.second.GetType() == NYT::TNode::Map) {
            ResolveTreeAttr(newPath, obj.second, dest);
        } else {
            dest.push_back(TResolvedNode(newPath, obj.second));
        }
    }
}

void ResolveTreeNode(NYT::IClientBasePtr client, const TString &path, const NYT::TNode &root, TDeque<TResolvedNode> &dest, const TDeque<TString> &reqAttrs) {
    if (root.GetType() == NYT::TNode::Map) {
        for (const auto &obj : root.AsMap()) {
            TDeque<TResolvedNode> attrs;
            if (obj.second.HasAttributes()) {
                ResolveTreeAttr("", obj.second.GetAttributes(), attrs);
            }

            const TString newPath = path + "/" + obj.first;
            TResolvedNode resolved(newPath, obj.second, attrs);

            if (obj.second.GetType() == NYT::TNode::Null) {
                bool needDiscover
                    = (resolved.HasAttr("/opaque") && IsTrue(resolved.GetAttr("/opaque")))
                    | (resolved.GetAttr("/type").AsString() == "link");

                if (needDiscover) {
                    ResolveTree(client, newPath, dest, reqAttrs);
                } else {
                    dest.push_back(resolved);
                }
            } else if (obj.second.GetType() == NYT::TNode::Map) {
                ResolveTreeNode(client, newPath, obj.second, dest, reqAttrs);
            }
        }
    }
}

bool ResolveTree(NYT::IClientBasePtr client, const TString &path, TDeque<TResolvedNode> &dest, const TDeque<TString> &reqAttrs) {
    NYT::TAttributeFilter filter;

    filter.AddAttribute("opaque");
    filter.AddAttribute("type");

    for (const TString &attr : reqAttrs) {
        filter.AddAttribute(attr);
    }

    NYT::TGetOptions opts;
    opts.AttributeFilter(filter);
    opts.MaxSize(Max<int>());

    NYT::TNode root;
    try {
        root = client->Get(path, opts);
    } catch(NYT::TErrorResponse& e) {
        return false;
    }

    TDeque<TResolvedNode> rootAttrs;
    if (root.HasAttributes()) {
        ResolveTreeAttr("", root.GetAttributes(), rootAttrs);
    }

    if (root.GetType() == NYT::TNode::Null) {
        TResolvedNode resolved(path, root, rootAttrs);

        bool needDiscover
            = (resolved.HasAttr("/opaque") && IsTrue(resolved.GetAttr("/opaque")))
            | (resolved.GetAttr("/type").AsString() == "link");

        if (needDiscover) {
            ResolveTree(client, path, dest, reqAttrs);
        } else {
            dest.push_back(resolved);
        }
    } else if (root.GetType() == NYT::TNode::Map) {
        ResolveTreeNode(client, path, root, dest, reqAttrs);
    }

    return true;
}

void SetArchiveAttr(NYT::IClientBasePtr client, const TString& table, const TString &eCodec, const TString &cCodec) {
    SetAttr(client, table, "compression_codec", cCodec);
    SetAttr(client, table, "erasure_codec", eCodec);
}

void UploadFile(NYT::IClientBasePtr client, const TString& srcPath, const TString& dstPath) {
    client->Create(dstPath, NYT::NT_FILE, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));
    TFileInput reader(srcPath);
    NYT::IFileWriterPtr writer = client->CreateFileWriter(dstPath);
    TransferData(&reader, writer.Get());
    writer->Finish();
}

} //namespace NYTUtils
} //namespace NWebmaster
