#include "device_cryptography_jni.h"

#include "jni_utils.h"

#include <yandex_io/android_sdk/cpp/launcher/global_context.h>
#include <library/cpp/jni/jni.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ble_decoder/ble_decoder.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/setup_parser/credentials.h>
#include <yandex_io/libs/setup_parser/setup_parser.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/hal/hal.h>
#include <yandex_io/protos/model_objects.pb.h>

YIO_DEFINE_LOG_MODULE("jni");

using namespace quasar;

namespace {

    jbyteArray convertBleCredentialsStatusToJbyteArray(JNIEnv* env, const proto::BleDecryptCredentialsStatus& protoCredentialsStatus) {
        int size = protoCredentialsStatus.ByteSize();
        std::vector<char> bytes(size);
        Y_PROTOBUF_SUPPRESS_NODISCARD protoCredentialsStatus.SerializeToArray(bytes.data(), size);

        jbyteArray byteArrayResult = env->NewByteArray(size);
        env->SetByteArrayRegion(byteArrayResult, 0, size, (jbyte*)&bytes[0]);

        return byteArrayResult;
    }

    quasar::proto::BleCredentials parseCredentialsToProto(const quasar::SetupParser::Credentials& credentials) {
        quasar::proto::BleCredentials protoCredentials;

        protoCredentials.set_is_initialized(credentials.isInitialized);
        protoCredentials.set_wifi_type(credentials.wifiType);

        for (const auto& ssid : credentials.SSIDs) {
            protoCredentials.add_ssids(TString(ssid));
        }

        protoCredentials.set_ssid_hash_code(credentials.SSIDHashCode);
        protoCredentials.set_password(TString(credentials.password));
        protoCredentials.set_x_token_code(TString(credentials.tokenCode));

        return protoCredentials;
    }

    quasar::proto::BleDecryptCredentialsStatus getBleDecryptCredentialsSuccess(const quasar::SetupParser::Credentials& credentials) {
        proto::BleDecryptCredentialsStatus protoCredentialsStatus;
        protoCredentialsStatus.set_status(proto::BleDecryptCredentialsStatus::SUCCESS);
        protoCredentialsStatus.mutable_ble_credentials()->CopyFrom(parseCredentialsToProto(credentials));

        return protoCredentialsStatus;
    }

    quasar::proto::BleDecryptCredentialsStatus getBleDecryptCredentialsError(const std::string& errorMessage) {
        quasar::proto::BleDecryptCredentialsStatus protoCredentialsStatus;
        protoCredentialsStatus.set_status(proto::BleDecryptCredentialsStatus::ERROR);
        protoCredentialsStatus.set_error_message(TString(errorMessage));

        return protoCredentialsStatus;
    }

    jbyteArray decryptBlePlainTextCredentials(JNIEnv* env, const std::vector<unsigned char>& plainText) {
        const std::string bytesString = bytesToString(plainText);
        const std::string digest = calcSHA1Digest(bytesString);
        auto device = GlobalContext::get().getDevice();

        YIO_LOG_DEBUG("Digest = " << digest);
        proto::BleDecryptCredentialsStatus result;
        try {
            const auto credentials = quasar::SetupParser::parseInitData(plainText);
            Json::Value event;
            event["networkType"] = "wifi";
            device->telemetry()->reportEvent("bleInitDataParsedSuccessfully", jsonToString(event));
            YIO_LOG_INFO("Successful parsing: " << credentials.toString());

            result = getBleDecryptCredentialsSuccess(credentials);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("FailedParseInitData", "std::exception & caught during parsing init data. " << e.what());
            Json::Value event;
            event["exception"] = e.what();

            device->telemetry()->reportEvent("bleInitDataParsingError", jsonToString(event));
            result = getBleDecryptCredentialsError(e.what());
        }

        return convertBleCredentialsStatusToJbyteArray(env, result);
    }
} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_DeviceCryptography7
 * Method:    doNativeBlePlainTextCredentials
 * Signature: (Ljava/lang/String;)[B;
 */
JNIEXPORT jbyteArray JNICALL Java_ru_yandex_io_sdk_jni_DeviceCryptography_doNativeDecryptBlePlainTextCredentials(JNIEnv* env, jobject /*obj*/, jstring jplainText)
{
    auto plainText = jstring_to_stdstring(env, jplainText);
    std::vector<unsigned char> plainTextChars(plainText.begin(), plainText.end());

    return decryptBlePlainTextCredentials(env, plainTextChars);
}

/*
 * Class:     ru_yandex_io_sdk_jni_DeviceCryptography
 * Method:    doNativeDecryptBleCredentials
 * Signature: ([B)[B;
 */
JNIEXPORT jbyteArray JNICALL Java_ru_yandex_io_sdk_jni_DeviceCryptography_doNativeDecryptBleCredentials(JNIEnv* env, jobject /*obj*/, jbyteArray bleCredentialsSerialized)
{
    TInputDataWrapper input(env, bleCredentialsSerialized);
    proto::EncryptedSetupCredentialsMessage protoEncryptedSetupCredentials;
    Y_PROTOBUF_SUPPRESS_NODISCARD protoEncryptedSetupCredentials.ParseFromArray(input.Bytes, input.Size);

    auto device = GlobalContext::get().getDevice();
    auto cryptographyConfig = getJson(device->configuration()->getServiceConfig("common"), "cryptography");
    auto deviceCryptography = device->hal()->createDeviceCryptography(cryptographyConfig);

    YIO_LOG_DEBUG("has_encrypted_setup_credentials_message");
    std::vector<unsigned char> decryptedCredentials;
    try {
        decryptedCredentials = decryptBleCredentials(protoEncryptedSetupCredentials, *deviceCryptography);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("FailedDecryptCredentials", "Failed to decrypt credentials: " << e.what());

        Json::Value event;
        event["exception"] = std::string(e.what());
        device->telemetry()->reportEvent("bleInitCredentialsDecryptFail", jsonToString(event));
        proto::BleDecryptCredentialsStatus result = getBleDecryptCredentialsError(e.what());

        return convertBleCredentialsStatusToJbyteArray(env, result);
    }

    return decryptBlePlainTextCredentials(env, decryptedCredentials);
}

/*
 * Class:     ru_yandex_io_sdk_jni_DeviceCryptography
 * Method:    doPrepareSignedHeaders
 * Signature: (Ljava/lang/String;)[I[Ljava/util/HashMap;)Z
 */
JNIEXPORT jobject JNICALL Java_ru_yandex_io_sdk_jni_DeviceCryptography_doPrepareSignedHeaders(JNIEnv* env, jobject /*obj*/, jstring jplainText)
{
    auto plainText = jstring_to_stdstring(env, jplainText);
    auto device = GlobalContext::get().getDevice();
    auto cryptographyConfig = getJson(device->configuration()->getServiceConfig("common"), "cryptography");
    auto deviceCryptography = device->hal()->createDeviceCryptography(cryptographyConfig);
    HttpClient::Headers headers = {};

    try {
        const auto signature = deviceCryptography->sign(plainText);
        HttpClient::addSignatureHeaders(headers, signature, deviceCryptography->getType());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("FailedPrepareSignedHeaders", "Can't prepare signed headers (signature version 2): " << e.what());
        // Don't fail here. Return empty headers
        YIO_LOG_WARN("Trying to register device without signature");
    }

    return std_string_string_map_to_jhashmap(env, headers);
}
