#include "jni_audio_client_event_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/audio_client_event_observer.h>
#include <yandex_io/sdk/converters/audio_client_event_converter.h>

#include <memory>
#include <string>
#include <vector>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperAudioClientEventObserver: public YandexIO::AudioClientEventObserver {
    public:
        JavaWrapperAudioClientEventObserver(JNIEnv* env, jobject instance)
            : audioClientEventObserverClass_(env->GetObjectClass(instance))
            , audioClientEventObserverInstance_(instance)
            , onAudioClientEventMethodID_(env->GetMethodID(audioClientEventObserverClass_.Get(),
                                                           "onAudioClientEvent", "([B)V"))
        {
        }

        ~JavaWrapperAudioClientEventObserver() {
            audioClientEventObserverClass_ = NJni::TGlobalClassRef();
            audioClientEventObserverInstance_ = NJni::TGlobalRef();
        }

        void onAudioClientEvent(const YandexIO::AudioClientEvent& audioClientEvent) override {
            auto protoEvent = YandexIO::convertAudioClientEventToProto(audioClientEvent);

            int size = protoEvent.ByteSize();
            std::vector<char> packedAudioClientEvent(size);

            Y_PROTOBUF_SUPPRESS_NODISCARD protoEvent.SerializeToArray(packedAudioClientEvent.data(), size);

            auto env = NJni::Env();
            auto jAudioClientEvent = env->NewByteArray(size);
            env->SetByteArrayRegion(jAudioClientEvent.Get(), 0, size, packedAudioClientEvent.data());

            env->CallVoidMethod(audioClientEventObserverInstance_.Get(), onAudioClientEventMethodID_, jAudioClientEvent.Get());
        }

    private:
        NJni::TGlobalClassRef audioClientEventObserverClass_;
        NJni::TGlobalRef audioClientEventObserverInstance_;

        jmethodID onAudioClientEventMethodID_;
    };

    std::shared_ptr<JavaWrapperAudioClientEventObserver> audioClientEventObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniAudioClientEventObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniAudioClientEventObserver_doRegister(JNIEnv* env, jobject instance) {
    if (audioClientEventObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "AudioClientEventObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Audio client event observer native/c++ registration");
    audioClientEventObserverWrapper = std::make_shared<JavaWrapperAudioClientEventObserver>(env, instance);

    YandexIO::getSDKSingleton()->addAudioClientEventObserver(audioClientEventObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniAudioClientEventObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniAudioClientEventObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    audioClientEventObserverWrapper.reset();
}
