#include "jni_auth_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/sdk/auth_observer.h>

#include <memory>
#include <string>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperAuthObserver: public YandexIO::AuthObserver {
    public:
        JavaWrapperAuthObserver(JNIEnv* env, jobject instance)
            : authObserverClass_(env->GetObjectClass(instance))
            , authObserverInstance_(instance)
            , onInvalidOAuthTokenMethodID_(env->GetMethodID(authObserverClass_.Get(),
                                                            "onInvalidOAuthToken", "(Ljava/lang/String;)V"))
            , onAuthenticationStatusMethodID_(env->GetMethodID(authObserverClass_.Get(),
                                                               "onAuthenticationStatus", "(Ljava/lang/String;ZLjava/lang/String;)V"))
            , onInvalidAuthenticationMethodID_(env->GetMethodID(authObserverClass_.Get(),
                                                                "onInvalidAuthentication", "(Ljava/lang/String;)V"))
        {
        }

        ~JavaWrapperAuthObserver() {
            authObserverClass_ = NJni::TGlobalClassRef();
            authObserverInstance_ = NJni::TGlobalRef();
        }

        void onInvalidOAuthToken(const std::string& token) override {
            auto env = NJni::Env();
            auto jToken = env->NewStringUTF(token);
            env->CallVoidMethod(authObserverInstance_.Get(), onInvalidOAuthTokenMethodID_, jToken.Get());
        }

        void onAuthenticationStatus(const std::string& oauthCode, bool isOk, const std::string& message) override {
            auto env = NJni::Env();
            auto jOauthCode = env->NewStringUTF(oauthCode);
            auto jMessage = env->NewStringUTF(message);
            env->CallVoidMethod(authObserverInstance_.Get(), onAuthenticationStatusMethodID_,
                                jOauthCode.Get(), (jboolean)isOk, jMessage.Get());
        }

        void onInvalidAuthentication(const std::string& uid) override {
            auto env = NJni::Env();
            auto jUid = env->NewStringUTF(uid);
            env->CallVoidMethod(authObserverInstance_.Get(), onInvalidAuthenticationMethodID_, jUid.Get());
        }

    private:
        NJni::TGlobalClassRef authObserverClass_;
        NJni::TGlobalRef authObserverInstance_;

        jmethodID onInvalidOAuthTokenMethodID_;
        jmethodID onAuthenticationStatusMethodID_;
        jmethodID onInvalidAuthenticationMethodID_;
    };

    std::shared_ptr<JavaWrapperAuthObserver> authObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniAuthObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniAuthObserver_doRegister(JNIEnv* env, jobject instance) {
    if (authObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "AuthObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Auth observer native/c++ registration");
    authObserverWrapper = std::make_shared<JavaWrapperAuthObserver>(env, instance);

    YandexIO::getSDKSingleton()->addAuthObserver(authObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniAuthObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniAuthObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    authObserverWrapper.reset();
}
