#include "jni_brick_status_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/brick_status_observer.h>

#include <memory>
#include <string>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperBrickStatusObserver: public YandexIO::BrickStatusObserver {
    public:
        JavaWrapperBrickStatusObserver(JNIEnv* env, jobject instance)
            : brickStatusObserverClass_(env->GetObjectClass(instance))
            , brickStatusObserverInstance_(instance)
            , onBrickStatusChangedMethodID_(env->GetMethodID(brickStatusObserverClass_.Get(),
                                                             "onBrickStatusChanged", "([B)V"))
        {
        }

        ~JavaWrapperBrickStatusObserver() {
            brickStatusObserverClass_ = NJni::TGlobalClassRef();
            brickStatusObserverInstance_ = NJni::TGlobalRef();
        }

        void onBrickStatusChanged(YandexIO::BrickStatus brickStatus, const std::string& brickStatusUrl, bool subscriptionMode) override {
            quasar::proto::BrickStatus quasarBrickStatus;
            switch (brickStatus) {
                case YandexIO::BrickStatus::BRICK:
                    quasarBrickStatus = quasar::proto::BrickStatus::BRICK;
                    break;
                case YandexIO::BrickStatus::NOT_BRICK:
                    quasarBrickStatus = quasar::proto::BrickStatus::NOT_BRICK;
                    break;
                case YandexIO::BrickStatus::UNKNOWN_BRICK_STATUS:
                    quasarBrickStatus = quasar::proto::BrickStatus::UNKNOWN_BRICK_STATUS;
                    break;
                case YandexIO::BrickStatus::BRICK_BY_TTL:
                    quasarBrickStatus = quasar::proto::BrickStatus::BRICK_BY_TTL;
                    break;
            }

            quasar::proto::BrickInfo brickInfo;
            brickInfo.set_brick_status(quasarBrickStatus);
            brickInfo.set_status_url(TString(brickStatusUrl));
            brickInfo.set_subscription_mode(subscriptionMode);

            YIO_LOG_DEBUG("Send brick status changed: " << brickInfo.DebugString());

            int size = brickInfo.ByteSize();
            std::vector<char> bytes(size);
            Y_PROTOBUF_SUPPRESS_NODISCARD brickInfo.SerializeToArray(bytes.data(), size);

            auto env = NJni::Env();
            auto jBytes = env->NewByteArray(size);
            env->SetByteArrayRegion(jBytes.Get(), 0, size, bytes.data());

            env->CallVoidMethod(brickStatusObserverInstance_.Get(), onBrickStatusChangedMethodID_, jBytes.Get());
        }

    private:
        NJni::TGlobalClassRef brickStatusObserverClass_;
        NJni::TGlobalRef brickStatusObserverInstance_;

        jmethodID onBrickStatusChangedMethodID_;
    };

    std::shared_ptr<JavaWrapperBrickStatusObserver> brickStatusObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniBrickStatusObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniBrickStatusObserver_doRegister(JNIEnv* env, jobject instance) {
    if (brickStatusObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "BrickStatusObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Brick status observer native/c++ registration");
    brickStatusObserverWrapper = std::make_shared<JavaWrapperBrickStatusObserver>(env, instance);

    YandexIO::getSDKSingleton()->addBrickStatusObserver(brickStatusObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniBrickStatusObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniBrickStatusObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    brickStatusObserverWrapper.reset();
}
