#include "jni_config_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/sdk/backend_config_observer.h>

#include <memory>
#include <string>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperConfigObserver: public YandexIO::BackendConfigObserver {
    public:
        JavaWrapperConfigObserver(JNIEnv* env, jobject instance)
            : configObserverClass_(env->GetObjectClass(instance))
            , configObserverInstance_(instance)
            , onConfigID_(env->GetMethodID(configObserverClass_.Get(), "onConfig",
                                           "(Ljava/lang/String;Ljava/lang/String;)V"))
        {
        }

        ~JavaWrapperConfigObserver() {
            configObserverClass_.Release();
            configObserverInstance_.Release();
        }

        void onSystemConfig(const std::string& configName, const std::string& configJsonValue) override {
            auto env = NJni::Env();
            auto jName = env->NewStringUTF(configName);
            auto jJsonValue = env->NewStringUTF(configJsonValue);

            YIO_LOG_DEBUG("Config observer received " << configName);
            env->CallVoidMethod(configObserverInstance_.Get(), onConfigID_, jName.Get(), jJsonValue.Get());
        }

        void onDeviceConfig(const std::string& configName, const std::string& configJsonValue) override {
            onSystemConfig(configName, configJsonValue);
        }

    private:
        NJni::TGlobalClassRef configObserverClass_;
        NJni::TGlobalRef configObserverInstance_;

        jmethodID onConfigID_;
    };

    std::shared_ptr<JavaWrapperConfigObserver> configObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniConfigObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniConfigObserver_doRegister(JNIEnv* env, jobject instance) {
    if (configObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "ConfigObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Config observer native/c++ registration");
    configObserverWrapper = std::make_shared<JavaWrapperConfigObserver>(env, instance);

    YandexIO::getSDKSingleton()->addBackendConfigObserver(configObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniConfigObserver
 * Method:    doSubscribeSystem
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniConfigObserver_doSubscribeSystem(JNIEnv* env, jobject /*obj*/, jstring configName) {
    const std::string configNameStr = jstring_to_stdstring(env, configName);

    YIO_LOG_DEBUG("Config observer native/c++ subscription to system [" << configNameStr << "]");
    YandexIO::getSDKSingleton()->subscribeToSystemConfig(configNameStr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniConfigObserver
 * Method:    doUnsubscribeSystem
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniConfigObserver_doUnsubscribeSystem(JNIEnv* env, jobject /*obj*/, jstring configName) {
    const std::string configNameStr = jstring_to_stdstring(env, configName);

    YIO_LOG_DEBUG("Config observer native/c++ unsubscription from system [" << configNameStr << "]");
    YandexIO::getSDKSingleton()->unsubscribeFromSystemConfig(configNameStr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniConfigObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniConfigObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    configObserverWrapper.reset();
}
