#include "jni_device_group_state_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/device_group_state.h>
#include <yandex_io/sdk/converters/device_group_state_converter.h>

#include <memory>
#include <string>
#include <vector>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperDeviceGroupStateObserver: public YandexIO::DeviceGroupStateObserver {
    public:
        JavaWrapperDeviceGroupStateObserver(JNIEnv* env, jobject instance)
            : deviceGroupStateObserverClass_(env->GetObjectClass(instance))
            , deviceGroupStateObserverInstance_(instance)
            , onDeviceGroupStateMethodID_(env->GetMethodID(deviceGroupStateObserverClass_.Get(),
                                                           "onDeviceGroupState", "([B)V"))
        {
        }

        ~JavaWrapperDeviceGroupStateObserver() {
            deviceGroupStateObserverClass_ = NJni::TGlobalClassRef();
            deviceGroupStateObserverInstance_ = NJni::TGlobalRef();
        }

        void onDeviceGroupState(const YandexIO::DeviceGroupState& deviceGroupState) override {
            auto protoEvent = YandexIO::convertDeviceGroupStateToProto(deviceGroupState);

            int size = protoEvent.ByteSize();
            std::vector<char> packedDeviceGroupState(size);

            Y_PROTOBUF_SUPPRESS_NODISCARD protoEvent.SerializeToArray(packedDeviceGroupState.data(), size);

            auto env = NJni::Env();
            auto jDeviceGroupState = env->NewByteArray(size);
            env->SetByteArrayRegion(jDeviceGroupState.Get(), 0, size, packedDeviceGroupState.data());

            env->CallVoidMethod(deviceGroupStateObserverInstance_.Get(), onDeviceGroupStateMethodID_, jDeviceGroupState.Get());
        }

    private:
        NJni::TGlobalClassRef deviceGroupStateObserverClass_;
        NJni::TGlobalRef deviceGroupStateObserverInstance_;

        jmethodID onDeviceGroupStateMethodID_;
    };

    std::shared_ptr<JavaWrapperDeviceGroupStateObserver> deviceGroupStateObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniDeviceGroupStateObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniDeviceGroupStateObserver_doRegister(JNIEnv* env, jobject instance) {
    if (deviceGroupStateObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "DeviceGroupStateObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Device group state observer native/c++ registration");
    deviceGroupStateObserverWrapper = std::make_shared<JavaWrapperDeviceGroupStateObserver>(env, instance);

    YandexIO::getSDKSingleton()->addDeviceGroupStateObserver(deviceGroupStateObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniDeviceGroupStateObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniDeviceGroupStateObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    deviceGroupStateObserverWrapper.reset();
}
