#include "jni_directive_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/sdk/directive_observer.h>

#include <memory>
#include <string>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperDirectiveObserver: public YandexIO::DirectiveObserver {
    public:
        JavaWrapperDirectiveObserver(JNIEnv* env, jobject instance)
            : directiveObserverClass_(env->GetObjectClass(instance))
            , directiveObserverInstance_(instance)
            , onDirectiveMethodID_(env->GetMethodID(directiveObserverClass_.Get(),
                                                    "onDirective", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V"))
        {
        }

        ~JavaWrapperDirectiveObserver() {
            directiveObserverClass_ = NJni::TGlobalClassRef();
            directiveObserverInstance_ = NJni::TGlobalRef();
        }

        void onDirective(const std::string& name, const std::string& vinsRequestId, const std::string& jsonPayload) override {
            YIO_LOG_DEBUG("report directive to listener in JNI: " << name << " (" << vinsRequestId << ") " << jsonPayload);

            auto env = NJni::Env();

            auto jName = env->NewStringUTF(name);
            auto jVinsRequestId = env->NewStringUTF(vinsRequestId);
            auto jJsonPayload = env->NewStringUTF(jsonPayload);

            env->CallVoidMethod(directiveObserverInstance_.Get(), onDirectiveMethodID_, jName.Get(), jVinsRequestId.Get(), jJsonPayload.Get());
        }

    private:
        NJni::TGlobalClassRef directiveObserverClass_;
        NJni::TGlobalRef directiveObserverInstance_;

        jmethodID onDirectiveMethodID_;
    };

    std::shared_ptr<JavaWrapperDirectiveObserver> directiveObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniDirectiveObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniDirectiveObserver_doRegister(JNIEnv* env, jobject instance) {
    if (directiveObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "DirectiveObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Directive observer native/c++ registration");
    directiveObserverWrapper = std::make_shared<JavaWrapperDirectiveObserver>(env, instance);

    YandexIO::getSDKSingleton()->addDirectiveObserver(directiveObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniDirectiveObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniDirectiveObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    directiveObserverWrapper.reset();
}
