#include "jni_glagol_discovery_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/glagol_sdk/i_discovery.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/converters/glagol_discovery_result_converter.h>

#include <memory>

using namespace quasar;

namespace {

    /**
     * Delegates onDiscoveryResult calls to attached Java object.
     */
    class JniGlagolDiscoveryObserver: public YandexIO::DiscoveryObserver {
    public:
        JniGlagolDiscoveryObserver(JNIEnv* env, jobject instance)
            : glagolDiscoveryObserverClass_(env->GetObjectClass(instance))
            , glagolDiscoveryObserverInstance_(instance)
            , onDiscoveryResultMethodID_(env->GetMethodID(glagolDiscoveryObserverClass_.Get(),
                                                          "onDiscoveryResult", "([B)V"))
        {
        }

        ~JniGlagolDiscoveryObserver() {
            glagolDiscoveryObserverClass_ = NJni::TGlobalClassRef();
            glagolDiscoveryObserverInstance_ = NJni::TGlobalRef();
        }

        void onDiscoveryResult(const glagol::IDiscovery::Result& res) override {
            YIO_LOG_DEBUG("report GlagolDiscoveryResult to listener in JNI");
            proto::GlagolDiscoveryResult protoResult = YandexIO::convertDiscoveryResultToProto(res);

            int protoStateSize = protoResult.ByteSize();
            std::vector<char> packedResult(protoStateSize);

            Y_PROTOBUF_SUPPRESS_NODISCARD protoResult.SerializeToArray(packedResult.data(), protoStateSize);

            auto env = NJni::Env();
            auto jDiscoveryResult = env->NewByteArray(protoStateSize);
            env->SetByteArrayRegion(jDiscoveryResult.Get(), 0, protoStateSize, packedResult.data());

            env->CallVoidMethod(glagolDiscoveryObserverInstance_.Get(), onDiscoveryResultMethodID_, jDiscoveryResult.Get());
        }

    private:
        NJni::TGlobalClassRef glagolDiscoveryObserverClass_;
        NJni::TGlobalRef glagolDiscoveryObserverInstance_;

        jmethodID onDiscoveryResultMethodID_;
    };

    std::shared_ptr<JniGlagolDiscoveryObserver> jniGlagolDiscoveryObserver;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniGlagolDiscoveryObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniGlagolDiscoveryObserver_doRegister(JNIEnv* env, jobject instance) {
    if (jniGlagolDiscoveryObserver) {
        ThrowJavaException(env, "java/lang/Exception", "GlagolDiscoveryObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("GlagolDiscoveryObserver native/c++ registration");
    jniGlagolDiscoveryObserver = std::make_shared<JniGlagolDiscoveryObserver>(env, instance);

    YandexIO::getSDKSingleton()->addDiscoveryObserver(jniGlagolDiscoveryObserver);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniGlagolDiscoveryObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniGlagolDiscoveryObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    jniGlagolDiscoveryObserver.reset();
}
