#include "jni_media_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/sdk/media_observer.h>

#include <memory>
#include <string>

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperMediaObserver: public YandexIO::MediaObserver {
    public:
        JavaWrapperMediaObserver(JNIEnv* env, jobject instance)
            : mediaObserverClass_(env->GetObjectClass(instance))
            , mediaObserverInstance_(instance)
            , onMediaErrorID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaError", "(I)V"))
            , onMediaRequestID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaRequest", "(I)V"))
            , onMediaStartedID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaStarted", "(I)V"))
            , onMediaPausedID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaPaused", "(I)V"))
            , onMediaResumedID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaResumed", "(I)V"))
            , onMediaSwitchedForwardID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaSwitchedForward", "(I)V"))
            , onMediaSwitchedBackwardID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaSwitchedBackward", "(I)V"))
            , onMediaLikedID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaLiked", "(I)V"))
            , onMediaDislikedID_(env->GetMethodID(mediaObserverClass_.Get(), "onMediaDisliked", "(I)V"))
        {
        }

        ~JavaWrapperMediaObserver() {
            mediaObserverClass_.Release();
            mediaObserverInstance_.Release();
        }

        void onMediaError(ContentType contentType) override {
            callJavaMethod(onMediaErrorID_, contentType);
        }

        void onMediaRequest(ContentType contentType) override {
            callJavaMethod(onMediaRequestID_, contentType);
        }

        void onMediaStarted(ContentType contentType) override {
            callJavaMethod(onMediaStartedID_, contentType);
        }

        void onMediaPaused(ContentType contentType) override {
            callJavaMethod(onMediaPausedID_, contentType);
        }

        void onMediaResumed(ContentType contentType) override {
            callJavaMethod(onMediaResumedID_, contentType);
        }

        void onMediaSwitchedForward(ContentType contentType) override {
            callJavaMethod(onMediaSwitchedForwardID_, contentType);
        }

        void onMediaSwitchedBackward(ContentType contentType) override {
            callJavaMethod(onMediaSwitchedBackwardID_, contentType);
        }

        void onMediaLiked(ContentType contentType) override {
            callJavaMethod(onMediaLikedID_, contentType);
        }

        void onMediaDisliked(ContentType contentType) override {
            callJavaMethod(onMediaDislikedID_, contentType);
        }

    private:
        void callJavaMethod(jmethodID methodId, ContentType contentType) {
            auto env = NJni::Env();
            auto jContentType = static_cast<jint>(contentType);
            env->CallVoidMethod(mediaObserverInstance_.Get(), methodId, jContentType);
        }

        NJni::TGlobalClassRef mediaObserverClass_;
        NJni::TGlobalRef mediaObserverInstance_;

        jmethodID onMediaErrorID_;
        jmethodID onMediaRequestID_;
        jmethodID onMediaStartedID_;
        jmethodID onMediaPausedID_;
        jmethodID onMediaResumedID_;
        jmethodID onMediaSwitchedForwardID_;
        jmethodID onMediaSwitchedBackwardID_;
        jmethodID onMediaLikedID_;
        jmethodID onMediaDislikedID_;
    };

    std::shared_ptr<JavaWrapperMediaObserver> mediaObserverWrapper;
} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniMediaObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniMediaObserver_doRegister(JNIEnv* env, jobject instance) {
    if (mediaObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "MediaObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Media observer native/c++ registration");
    mediaObserverWrapper = std::make_shared<JavaWrapperMediaObserver>(env, instance);

    YandexIO::getSDKSingleton()->addMediaObserver(mediaObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniMediaObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniMediaObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    mediaObserverWrapper.reset();
}
