#include "jni_music_state_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/music_state_observer.h>

#include <memory>
#include <string>
#include <vector>

using namespace quasar;

namespace {

    proto::AppState::MusicState::TrackType convertTrackType(const YandexIO::MusicStateObserver::TrackType type) {
        switch (type) {
            case YandexIO::MusicStateObserver::TrackType::MUSIC:
                return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_MUSIC;
            case YandexIO::MusicStateObserver::TrackType::FAIRY_TALE:
                return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_FAIRY_TALE;
            case YandexIO::MusicStateObserver::TrackType::PODCAST:
                return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_PODCAST;
            case YandexIO::MusicStateObserver::TrackType::AUDIOBOOK:
                return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_AUDIOBOOK;
            case YandexIO::MusicStateObserver::TrackType::UNKNOWN:
                return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_UNKNOWN;
        }
        return proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_UNKNOWN;
    }

    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperMusicStateObserver: public YandexIO::MusicStateObserver {
    public:
        JavaWrapperMusicStateObserver(JNIEnv* env, jobject instance)
            : musicStateObserverClass_(env->GetObjectClass(instance))
            , musicStateObserverInstance_(instance)
            , onMusicStateChangedMethodID_(env->GetMethodID(musicStateObserverClass_.Get(),
                                                            "onMusicStateChanged", "([B)V"))
        {
        }

        ~JavaWrapperMusicStateObserver() {
            musicStateObserverClass_ = NJni::TGlobalClassRef();
            musicStateObserverInstance_ = NJni::TGlobalRef();
        }

        void onMusicStateChanged(const MusicState& musicState) override {
            quasar::proto::AppState::MusicState protoState;
            protoState.set_is_paused(musicState.isPaused);
            protoState.set_title(TString(musicState.title));
            protoState.set_artists(TString(musicState.artists));
            protoState.set_cover_uri(TString(musicState.coverUri));
            protoState.set_duration_ms(musicState.durationMs);
            protoState.set_progress(musicState.progress);
            protoState.set_next_track_type(convertTrackType(musicState.nextTrackType));

            int protoStateSize = protoState.ByteSize();
            std::vector<char> packedMusicState(protoStateSize);

            Y_PROTOBUF_SUPPRESS_NODISCARD protoState.SerializeToArray(packedMusicState.data(), protoStateSize);

            auto env = NJni::Env();
            auto jMusicState = env->NewByteArray(protoStateSize);
            env->SetByteArrayRegion(jMusicState.Get(), 0, protoStateSize, packedMusicState.data());

            env->CallVoidMethod(musicStateObserverInstance_.Get(), onMusicStateChangedMethodID_, jMusicState.Get());
        }

    private:
        NJni::TGlobalClassRef musicStateObserverClass_;
        NJni::TGlobalRef musicStateObserverInstance_;

        jmethodID onMusicStateChangedMethodID_;
    };

    std::shared_ptr<JavaWrapperMusicStateObserver> musicStateObserverWrapper;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniMusicStateObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniMusicStateObserver_doRegister(JNIEnv* env, jobject instance) {
    if (musicStateObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "MusicStateObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("Music state observer native/c++ registration");
    musicStateObserverWrapper = std::make_shared<JavaWrapperMusicStateObserver>(env, instance);

    YandexIO::getSDKSingleton()->addMusicStateObserver(musicStateObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniMusicStateObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniMusicStateObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    musicStateObserverWrapper.reset();
}
