#include "jni_sdk_state_observer_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/yandex_io.pb.h>
#include <yandex_io/sdk/sdk_state.h>
#include <yandex_io/sdk/sdk_state_observer.h>

#include <memory>
#include <string>

using namespace quasar;
using namespace YandexIO;

namespace {

    proto::AliceState convertAliceState(const SDKState::AliceState& aliceState)
    {
        proto::AliceState resultState;
        switch (aliceState.state) {
            case SDKState::AliceState::State::IDLE:
                resultState.set_state(proto::AliceState_State::AliceState_State_IDLE);
                break;
            case SDKState::AliceState::State::LISTENING:
                resultState.set_state(proto::AliceState_State::AliceState_State_LISTENING);
                break;
            case SDKState::AliceState::State::SPEAKING:
                resultState.set_state(proto::AliceState_State::AliceState_State_SPEAKING);
                break;
            case SDKState::AliceState::State::THINKING:
                resultState.set_state(proto::AliceState_State::AliceState_State_BUSY);
                break;
            case SDKState::AliceState::State::SHAZAM:
                resultState.set_state(proto::AliceState_State::AliceState_State_SHAZAM);
                break;
            default:
            case SDKState::AliceState::State::NONE:
                resultState.set_state(proto::AliceState_State::AliceState_State_NONE);
                break;
        }

        if (aliceState.recognizedPhrase) {
            resultState.set_recognized_phrase(TString(*aliceState.recognizedPhrase));
        }

        if (aliceState.vinsResponse) {
            auto vinsResponse = resultState.mutable_vins_response();
            if (aliceState.vinsResponse->outputSpeech) {
                vinsResponse->set_output_speech(TString(*aliceState.vinsResponse->outputSpeech));
            }
            if (aliceState.vinsResponse->textOnly) {
                vinsResponse->set_text_only(*aliceState.vinsResponse->textOnly);
            }
            if (aliceState.vinsResponse->divCard) {
                vinsResponse->set_div_card(TString(*aliceState.vinsResponse->divCard));
            }
            vinsResponse->mutable_suggests()->Reserve(aliceState.vinsResponse->suggests.size());
            for (const auto& suggest : aliceState.vinsResponse->suggests) {
                proto::VinsResponse::Suggest* protoSuggest = vinsResponse->add_suggests();
                protoSuggest->set_text(TString(suggest.text));
            }
        }

        return resultState;
    }

    proto::IOEvent_SDKState_NotificationState convertNotificationState(SDKState::NotificationState notificationState)
    {
        if (notificationState == SDKState::NotificationState::AVAILABLE) {
            return quasar::proto::IOEvent_SDKState_NotificationState_AVAILABLE;
        }
        if (notificationState == SDKState::NotificationState::PASSIVE) {
            return quasar::proto::IOEvent_SDKState_NotificationState_PASSIVE;
        }

        return quasar::proto::IOEvent_SDKState_NotificationState_NONE;
    }

    proto::ConfigurationState convertConfigurationState(SDKState::ConfigurationState configurationState)
    {
        switch (configurationState) {
            case SDKState::ConfigurationState::UNKNOWN:
                return proto::ConfigurationState::UNKNOWN_STATE;
            case SDKState::ConfigurationState::CONFIGURING:
                return proto::ConfigurationState::CONFIGURING;
            case SDKState::ConfigurationState::CONFIGURED:
                return proto::ConfigurationState::CONFIGURED;
        }
        return proto::ConfigurationState::UNKNOWN_STATE;
    }

    proto::WifiStatus convertWifiState(const SDKState::WifiState& wifiStatus)
    {
        proto::WifiStatus protoWifiStatus;
        protoWifiStatus.set_status(wifiStatus.isWifiConnected ? proto::WifiStatus::CONNECTED : proto::WifiStatus::NOT_CONNECTED);
        protoWifiStatus.set_internet_reachable(wifiStatus.isInternetReachable);
        return protoWifiStatus;
    }

    proto::CalldSessionState convertCallState(const SDKState::CallState& calldState) {
        proto::CalldSessionState calldSessionState;

        switch (calldState.status) {
            case SDKState::CallState::Status::NEW:
                calldSessionState.set_status(proto::CalldSessionState::NEW);
                break;

            case SDKState::CallState::Status::DIALING:
                calldSessionState.set_status(proto::CalldSessionState::DIALING);
                break;

            case SDKState::CallState::Status::RINGING:
                calldSessionState.set_status(proto::CalldSessionState::RINGING);
                break;

            case SDKState::CallState::Status::ACCEPTING:
                calldSessionState.set_status(proto::CalldSessionState::ACCEPTING);
                break;

            case SDKState::CallState::Status::CONNECTING:
                calldSessionState.set_status(proto::CalldSessionState::CONNECTING);
                break;

            case SDKState::CallState::Status::CONNECTED:
                calldSessionState.set_status(proto::CalldSessionState::CONNECTED);
                break;

            case SDKState::CallState::Status::ENDED_OK:
                calldSessionState.set_status(proto::CalldSessionState::ENDED);
                break;

            case SDKState::CallState::Status::ENDED_FAILURE:
                calldSessionState.set_status(proto::CalldSessionState::ENDED);
                calldSessionState.mutable_ended_with_error();
                break;

            case SDKState::CallState::Status::NOCALL:
                calldSessionState.set_status(proto::CalldSessionState::NOCALL);
                break;
        }

        switch (calldState.direction) {
            case SDKState::CallState::Direction::INCOMING:
                calldSessionState.set_direction(proto::CalldSessionState::INCOMING);
                break;

            case SDKState::CallState::Direction::OUTGOING:
                calldSessionState.set_direction(proto::CalldSessionState::OUTGOING);
                break;
        }

        return calldSessionState;
    }

    proto::IOEvent::NtpdState convertNtpdState(const SDKState::NtpdState& ntpdState) {
        proto::IOEvent::NtpdState protoNtpdState;
        if (ntpdState.clockSynchronized.has_value()) {
            protoNtpdState.set_clock_synchronized(*ntpdState.clockSynchronized);
        }
        return protoNtpdState;
    }

    proto::UpdateState convertUpdateState(const SDKState::UpdateState& updateState)
    {
        proto::UpdateState protoState;
        protoState.set_download_progress(updateState.downloadProgress);
        protoState.set_is_critical(updateState.isCritical);
        switch (updateState.state) {
            case SDKState::UpdateState::State::DOWNLOADING: {
                protoState.set_state(proto::UpdateState_State_DOWNLOADING);
                break;
            }
            case SDKState::UpdateState::State::APPLYING: {
                protoState.set_state(proto::UpdateState_State_APPLYING);
                break;
            }
            default:
            case SDKState::UpdateState::State::NONE: {
                protoState.set_state(proto::UpdateState_State_NONE);
                break;
            }
        }
        return protoState;
    }

    proto::AudioClientEvent convertAudioState(const SDKState::PlayerState::Audio& audioState) {
        proto::AudioClientEvent event;
        if (audioState.isPlaying) {
            event.set_state(proto::AudioClientState::PLAYING);
        }
        return event;
    }

    proto::AppState::MusicState convertMusicState(const SDKState::PlayerState::Music& musicState) {
        proto::AppState::MusicState ms;
        ms.set_current_track_id(TString(musicState.currentTrackId));
        ms.set_is_paused(!musicState.isPlaying);
        ms.set_title(TString(musicState.title));
        ms.set_artists(TString(musicState.artists));
        ms.set_cover_uri(TString(musicState.coverURL));
        return ms;
    }

    proto::AppState::RadioState convertRadioState(const SDKState::PlayerState::Radio& radioState) {
        proto::AppState::RadioState rs;
        rs.set_is_paused(!radioState.isPlaying);
        return rs;
    }

    proto::AppState::ScreenState convertScreenState(const SDKState::ScreenState& screenState)
    {
        proto::AppState::ScreenState protoState;
        protoState.set_is_screensaver_on(screenState.isScreensaverOn);
        return protoState;
    }

    ::google::protobuf::RepeatedPtrField<proto::IOEvent::TimersTimingsState> convertTimersState(const std::vector<YandexIO::SDKState::TimerState>& timers) {
        ::google::protobuf::RepeatedPtrField<proto::IOEvent::TimersTimingsState> result;
        result.Reserve(timers.size());
        for (const auto& [id, start, end] : timers) {
            const auto timersStartMs = std::chrono::duration_cast<std::chrono::milliseconds>(start.time_since_epoch());
            const auto timersEndMs = std::chrono::duration_cast<std::chrono::milliseconds>(end.time_since_epoch());
            auto item = result.Add();
            item->set_start_timer_ms(timersStartMs.count());
            item->set_end_timer_ms(timersEndMs.count());
            item->set_id(TString(id));
        }
        return result;
    }

    proto::IotState convertIotState(SDKState::IotState iotState)
    {
        proto::IotState protoState;
        static_assert(proto::IotState::State_MAX == 2, "There must be 3 IOT states");
        switch (iotState) {
            case SDKState::IotState::IDLE:
                protoState.set_current_state(proto::IotState::IDLE);
                break;
            case SDKState::IotState::STARTING_DISCOVERY:
                protoState.set_current_state(proto::IotState::STARTING_DISCOVERY);
                break;
            case SDKState::IotState::DISCOVERY_IN_PROGRESS:
                protoState.set_current_state(proto::IotState::DISCOVERY_IN_PROGRESS);
                break;
            default:
                break;
        }
        return protoState;
    }

    proto::MultiroomState convertMultiroomState(const SDKState::MultiroomState& multiroomState)
    {
        proto::MultiroomState state;

        switch (multiroomState.mode) {
            default:
            case SDKState::MultiroomState::Mode::NONE:
                state.set_mode(proto::MultiroomState::NONE);
                break;
            case SDKState::MultiroomState::Mode::MASTER:
                state.set_mode(proto::MultiroomState::MASTER);
                break;
            case SDKState::MultiroomState::Mode::SLAVE:
                state.set_mode(proto::MultiroomState::SLAVE);
                break;
        }
        state.mutable_multiroom_broadcast()->set_state(multiroomState.playing
                                                           ? proto::MultiroomBroadcast::PLAYING
                                                           : proto::MultiroomBroadcast::NONE);
        state.set_slave_clock_syncing(multiroomState.slaveClockSyncing);
        switch (multiroomState.slaveSyncLevel) {
            case SDKState::MultiroomState::SyncLevel::NONE:
                state.set_slave_sync_level(proto::MultiroomState::NOSYNC);
                break;
            case SDKState::MultiroomState::SyncLevel::WEAK:
                state.set_slave_sync_level(proto::MultiroomState::WEAK);
                break;
            case SDKState::MultiroomState::SyncLevel::STRONG:
                state.set_slave_sync_level(proto::MultiroomState::STRONG);
                break;
            default:
                break;
        }
        return state;
    }

    proto::AllStartupSettings convertAllStartupSettings(const SDKState::AllStartupSettings& allStartupSettings)
    {
        proto::AllStartupSettings protoState;

        protoState.set_all_startup_info(allStartupSettings.allStartupInfo);
        protoState.set_auth_token(TString(allStartupSettings.authToken));
        protoState.set_passport_uid(TString(allStartupSettings.passportUid));

        return protoState;
    }

    // TODO[eninng]: rename to SDKState...
    /**
     * Delegates all calls to attached Java object.
     */
    class JavaWrapperSDKStateObserver: public YandexIO::SDKStateObserver {
    public:
        JavaWrapperSDKStateObserver(JNIEnv* env, jobject instance)
            : sdkStateObserverClass_(env->GetObjectClass(instance))
            , sdkStateObserverInstance_(instance)
            , onSDKStateChangedMethodID_(env->GetMethodID(sdkStateObserverClass_.Get(),
                                                          "onSDKStateChanged", "([B)V"))
        {
        }

        ~JavaWrapperSDKStateObserver() {
            sdkStateObserverClass_ = NJni::TGlobalClassRef();
            sdkStateObserverInstance_ = NJni::TGlobalRef();
        }

        void onSDKState(const YandexIO::SDKState& state) override {
            proto::IOEvent::SDKState protoState;

            protoState.mutable_alice_state()->CopyFrom(convertAliceState(state.aliceState));
            protoState.mutable_alarms_state()->set_alarm_playing(state.isAlarmPlaying);
            protoState.mutable_alarms_state()->set_timer_playing(state.isTimerPlaying);
            protoState.set_icalendar_state(TString(state.iCalendarState));
            protoState.mutable_timers_timings()->CopyFrom(convertTimersState(state.timers));
            protoState.mutable_reminder_state()->set_reminder_playing(state.isReminderPlaying);
            protoState.set_configuration_state(convertConfigurationState(state.configurationState));
            protoState.mutable_update_state()->CopyFrom(convertUpdateState(state.updateState));
            protoState.mutable_wifi_status()->CopyFrom(convertWifiState(state.wifiState));
            protoState.set_notification_state(convertNotificationState(state.notificationState));
            protoState.mutable_app_state()->mutable_audio_player_state()->CopyFrom(convertAudioState(state.playerState.audio));
            protoState.mutable_app_state()->mutable_music_state()->CopyFrom(convertMusicState(state.playerState.music));
            protoState.mutable_app_state()->mutable_radio_state()->CopyFrom(convertRadioState(state.playerState.radio));
            protoState.mutable_app_state()->mutable_screen_state()->CopyFrom(convertScreenState(state.screenState));
            protoState.mutable_calld_state()->CopyFrom(convertCallState(state.callState));
            protoState.mutable_ntpd_state()->CopyFrom(convertNtpdState(state.ntpdState));
            protoState.mutable_iot_state()->CopyFrom(convertIotState(state.iotState));
            protoState.mutable_multiroom_state()->CopyFrom(convertMultiroomState(state.multiroomState));
            protoState.mutable_all_startup_settings()->CopyFrom(convertAllStartupSettings(state.allStartupSettings));

            int protoStateSize = protoState.ByteSize();
            std::vector<char> packedSDKState(protoStateSize);

            Y_PROTOBUF_SUPPRESS_NODISCARD protoState.SerializeToArray(packedSDKState.data(), protoStateSize);

            auto env = NJni::Env();
            auto jSDKState = env->NewByteArray(protoStateSize);
            env->SetByteArrayRegion(jSDKState.Get(), 0, protoStateSize, packedSDKState.data());

            env->CallVoidMethod(sdkStateObserverInstance_.Get(), onSDKStateChangedMethodID_, jSDKState.Get());
        }

    private:
        NJni::TGlobalClassRef sdkStateObserverClass_;
        NJni::TGlobalRef sdkStateObserverInstance_;

        jmethodID onSDKStateChangedMethodID_;
    };

    std::shared_ptr<JavaWrapperSDKStateObserver> sdkStateObserverWrapper;
} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_JniSDKStateObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniSDKStateObserver_doRegister(JNIEnv* env, jobject instance) {
    if (sdkStateObserverWrapper) {
        ThrowJavaException(env, "java/lang/Exception", "SDKStateObserver is already registered");
        return;
    }

    YIO_LOG_DEBUG("SDKState observer native/c++ registration");
    sdkStateObserverWrapper = std::make_shared<JavaWrapperSDKStateObserver>(env, instance);

    YandexIO::getSDKSingleton()->addSDKStateObserver(sdkStateObserverWrapper);
}

/*
 * Class:     ru_yandex_io_sdk_jni_JniSDKStateObserver
 * Method:    doReset
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_JniSDKStateObserver_doReset(JNIEnv* /*env*/, jobject /*obj*/) {
    sdkStateObserverWrapper.reset();
}
