#include "jni_utils.h"

#include <util/generic/scope.h>

#include <map>

using namespace quasar;

TInputDataWrapper::TInputDataWrapper(JNIEnv* env, jbyteArray input)
    : Env(env)
    , Input(input)
    , Bytes(Env->GetByteArrayElements(Input, nullptr))
    , Size(Env->GetArrayLength(Input))
{
}

TInputDataWrapper::~TInputDataWrapper() {
    Env->ReleaseByteArrayElements(Input, Bytes, 0);
}

void quasar::ThrowJavaException(JNIEnv* env, const char* cls, const char* message) {
    jclass je = env->FindClass(cls);
    env->ThrowNew(je, message);
}

std::string quasar::jstring_to_stdstring(JNIEnv* env, jstring jStr) {
    if (!jStr) {
        return "";
    }

    const jclass stringClass = env->GetObjectClass(jStr);
    Y_DEFER {
        env->DeleteLocalRef(stringClass);
    };

    const jmethodID getBytes = env->GetMethodID(stringClass, "getBytes", "(Ljava/lang/String;)[B");
    const jbyteArray stringJbytes = (jbyteArray)env->CallObjectMethod(jStr, getBytes, env->NewStringUTF("UTF-8"));
    Y_DEFER {
        env->DeleteLocalRef(stringJbytes);
    };

    const TInputDataWrapper wrapper(env, stringJbytes);
    return std::string((char*)wrapper.Bytes, wrapper.Size);
}

jobject quasar::std_string_string_map_to_jhashmap(JNIEnv* env, const std::map<std::string, std::string>& map)
{
    jclass mapClass = env->FindClass("java/util/HashMap");
    if (mapClass == NULL)
        return NULL;

    jmethodID init = env->GetMethodID(mapClass, "<init>", "()V");
    jobject hashMap = env->NewObject(mapClass, init);
    jmethodID put = env->GetMethodID(mapClass, "put", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");

    std::map<std::string, std::string>::const_iterator citr = map.begin();
    for (; citr != map.end(); ++citr) {
        jstring keyJava = env->NewStringUTF(citr->first.c_str());
        jstring valueJava = env->NewStringUTF(citr->second.c_str());

        env->CallObjectMethod(hashMap, put, keyJava, valueJava);

        Y_DEFER {
            env->DeleteLocalRef(keyJava);
        };
        Y_DEFER {
            env->DeleteLocalRef(valueJava);
        };
    }

    jobject hashMapGlobal = static_cast<jobject>(env->NewGlobalRef(hashMap));
    Y_DEFER {
        env->DeleteLocalRef(hashMap);
    };
    Y_DEFER {
        env->DeleteLocalRef(mapClass);
    };

    return hashMapGlobal;
}
