#include "mdns_observer_jni.h"

#include <yandex_io/android_sdk/cpp/launcher/global_context.h>
#include <library/cpp/jni/jni.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/mdns/nsd_messager.h>
#include <yandex_io/libs/mdns/nsd_receiver.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/services/glagold/resolve_handler.h>

#include <cstdint>
#include <string>

namespace {
    class JniNsdMessager: public quasar::INsdMessager {
    public:
        JniNsdMessager(JNIEnv* env, jobject instance)
            : mdnsObserverClass_{env->GetObjectClass(instance)}
            , mdnsObserverInstance_{instance}
            , onNsdEnableMethodID_{env->GetMethodID(mdnsObserverClass_.Get(), "onNsdEnable", "(ZIII)V")}
            , onNsdDisableMethodID_{env->GetMethodID(mdnsObserverClass_.Get(), "onNsdDisable", "()V")}
        {
        }

        static uint32_t optBoolToI(OptBool src) {
            if (src) {
                return src.value() ? 2u : 1u;
            }
            return 0;
        }

        void enableNsd(bool guestMode, uint32_t port, OptBool stereopair, OptBool tandem) override {
            auto env = NJni::Env();
            env->CallVoidMethod(mdnsObserverInstance_.Get(), onNsdEnableMethodID_, guestMode, port, optBoolToI(stereopair), optBoolToI(tandem));
        }

        void disableNsd() override {
            auto env = NJni::Env();
            env->CallVoidMethod(mdnsObserverInstance_.Get(), onNsdDisableMethodID_);
        }

    private:
        NJni::TGlobalClassRef mdnsObserverClass_;
        NJni::TGlobalRef mdnsObserverInstance_;

        jmethodID onNsdEnableMethodID_;
        jmethodID onNsdDisableMethodID_;
    };

    std::string jbyteArrayToString(JNIEnv* env, jbyteArray array) {
        const size_t len = static_cast<size_t>(env->GetArrayLength(array));
        std::string buffer(len, ' ');
        env->GetByteArrayRegion(array, 0, len, reinterpret_cast<jbyte*>(&buffer[0]));
        return buffer;
    }

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_MdnsObserver
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_MdnsObserver_doRegister(
    JNIEnv* env,
    jobject instance) {
    YIO_LOG_DEBUG("MdnsObserver native/c++ registration");
    auto jniNsdMessager = std::make_unique<JniNsdMessager>(
        env,
        instance);
    quasar::GlobalContext::get().getNsdMessagerFactory()->setJniNsdMessager(std::move(jniNsdMessager));
}

/*
 * Class:     ru_yandex_io_sdk_jni_MdnsObserver
 * Method:    doUnregister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_MdnsObserver_doUnregister(
    JNIEnv* /*jenv*/,
    jobject /*obj*/) {
    quasar::GlobalContext::get().setNsdMessagerFactory(nullptr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_MdnsObserver
 * Method:    doOnHostDiscovered
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_MdnsObserver_doOnHostDiscovered(
    JNIEnv* env,
    jobject /*obj*/,
    jbyteArray serializedNsdInfo) {
    const TString buffer = jbyteArrayToString(env, serializedNsdInfo);
    quasar::proto::NsdInfo nsdInfo;
    Y_PROTOBUF_SUPPRESS_NODISCARD nsdInfo.ParseFromString(buffer);
    quasar::GlobalContext::get().getNsdMessagerFactory()->nsdReceiver().onNsdDiscovered(nsdInfo);
}

/*
 * Class:     ru_yandex_io_sdk_jni_MdnsObserver
 * Method:    doOnHostListDiscovered
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_MdnsObserver_doOnHostListDiscovered(
    JNIEnv* env,
    jobject /*obj*/,
    jbyteArray serializedNsdInfoList) {
    const TString buffer = jbyteArrayToString(env, serializedNsdInfoList);
    quasar::proto::NsdInfoList nsdInfo;
    Y_PROTOBUF_SUPPRESS_NODISCARD nsdInfo.ParseFromString(buffer);
    quasar::GlobalContext::get().getNsdMessagerFactory()->nsdReceiver().onNsdListDiscovered(nsdInfo);
}

/*
 * Class:     ru_yandex_io_sdk_jni_MdnsObserver
 * Method:    doOnHostLost
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_MdnsObserver_doOnHostLost(
    JNIEnv* env,
    jobject /*obj*/,
    jbyteArray serializedNsdInfo) {
    const TString buffer = jbyteArrayToString(env, serializedNsdInfo);
    quasar::proto::NsdInfo nsdInfo;
    Y_PROTOBUF_SUPPRESS_NODISCARD nsdInfo.ParseFromString(buffer);
    quasar::GlobalContext::get().getNsdMessagerFactory()->nsdReceiver().onNsdLost(nsdInfo);
}
