#include "quasar_client_jni.h"

#include "jni_utils.h"

#include <yandex_io/android_sdk/cpp/launcher/global_context.h>

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <google/protobuf/text_format.h>

using namespace quasar;

namespace {

    void printMessage(const ipc::SharedMessage& message) {
        TString textFormat;
        google::protobuf::TextFormat::PrintToString(*message, &textFormat);

        YIO_LOG_INFO("----------------------Received---------------------"
                     << '\n'
                     << textFormat
                     << '\n'
                     << "---------------------------------------------------"
                     << std::endl);
    }

    std::shared_ptr<ipc::IConnector> createIpcConnector(const std::string& serviceName) {
        auto connector = GlobalContext::get().getIpcFactory()->createIpcConnector(serviceName);

        connector->setMessageHandler(printMessage);
        connector->setConnectHandler([serviceName]() {
            YIO_LOG_INFO("Connected to " << serviceName);
        });
        connector->setDisconnectHandler([serviceName]() {
            YIO_LOG_INFO("Disconnected from " << serviceName);
        });
        connector->setConnectionErrorHandler([](const std::string& error) {
            YIO_LOG_WARN("Connection error: " << error);
        });

        connector->connectToService();
        connector->waitUntilConnected();

        return connector;
    }

    void sendQuasarMessage(const TString& serviceName, const TString& quasarMessage) {
        auto connector = createIpcConnector(serviceName);

        proto::QuasarMessage message;
        google::protobuf::TextFormat::ParseFromString(quasarMessage, &message);
        YIO_LOG_INFO("Sending to: "
                     << serviceName
                     << '\n'
                     << quasarMessage
                     << "\n---------------------------------------------------"
                     << std::endl);

        auto result = connector->sendMessage(std::move(message));
        YIO_LOG_INFO("Send result: " << result);
        connector->shutdown();
    }

} // namespace

/*
 * Class:     ru_yandex_quasar_android_app_QuasarClient
 * Method:    doNativeRun
 * Signature: (Ljava/lang/String;ILjava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_QuasarClient_sendQuasarMessage(JNIEnv* env, jobject /*obj*/, jstring serviceName, jstring quasarMessage) {
    try {
        sendQuasarMessage(jstring_to_stdstring(env, serviceName), jstring_to_stdstring(env, quasarMessage));
    } catch (...) {
        ThrowJavaException(env, "java/lang/Exception", CurrentExceptionMessage().c_str());
    }
}
