#include "volume_manager_jni.h"

#include "jni_utils.h"

#include <yandex_io/android_sdk/cpp/launcher/global_context.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>
#include <yandex_io/android_sdk/cpp/services/android_volume_manager.h>

#include <yandex_io/capabilities/volume/volume_capability.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <library/cpp/jni/jni.h>

#include <stdint.h>

using namespace quasar;

namespace {

    struct VolumeManagerGlobalContext {
        NJni::TGlobalClassRef volumeManagerClass;
        NJni::TGlobalRef volumeManagerInstance;
        std::shared_ptr<AndroidVolumeManager> volumeManagerNative;
        std::shared_ptr<YandexIO::IVolumeCapability> volumeCapability;
    };

    VolumeManagerGlobalContext context;

} // namespace

/*
 * Class:     ru_yandex_io_sdk_jni_VolumeManager
 * Method:    doRegister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_VolumeManager_doRegister(JNIEnv* env, jobject instance) {
    if (context.volumeManagerClass || context.volumeManagerInstance) {
        ThrowJavaException(env, "java/lang/Exception", "VolumeManager is already registered");
        return;
    }

    YIO_LOG_DEBUG("Volume manager native/c++ registration");

    jclass clz = env->GetObjectClass(instance);
    context.volumeManagerClass = NJni::TGlobalClassRef(clz);
    context.volumeManagerInstance = NJni::TGlobalRef(instance);

    context.volumeManagerNative = std::make_shared<AndroidVolumeManager>(
        GlobalContext::get().getDevice(),
        YandexIO::getSDKSingleton(),
        /* getVolumeCallback = */ [] {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "getVolume", "()I", false);
            return envi->CallIntMethod(context.volumeManagerInstance.Get(), methodId); },
        /* setVolumeCallback = */ [](int androidVolume) {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "setVolume", "(I)V", false);
            envi->CallVoidMethod(context.volumeManagerInstance.Get(), methodId, static_cast<jint>(androidVolume)); },
        /* adjustVolumeCallback = */ [](AndroidVolumeManager::AdjustVolumeDirection androidDirection) {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "adjustVolume", "(I)V", false);
            envi->CallVoidMethod(context.volumeManagerInstance.Get(), methodId, static_cast<jint>(androidDirection)); },
        /* isMutedCallback = */ [] {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "isMuted", "()Z", false);
            return envi->CallBooleanMethod(context.volumeManagerInstance.Get(), methodId); },
        /* muteCallback = */ []() {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "mute", "()V", false);
            envi->CallVoidMethod(context.volumeManagerInstance.Get(), methodId); },
        /* unmuteCallback = */ []() {
            auto envi = NJni::Env();
            auto methodId = envi->GetMethodID(context.volumeManagerClass.Get(), "unmute", "()V", false);
            envi->CallVoidMethod(context.volumeManagerInstance.Get(), methodId); });
    context.volumeManagerNative->start();

    context.volumeCapability = std::make_shared<YandexIO::VolumeCapability>(context.volumeManagerNative);
    YandexIO::getSDKSingleton()->getEndpointStorage()->getLocalEndpoint()->addCapability(context.volumeCapability);
}

/*
 * Class:     ru_yandex_io_sdk_jni_VolumeManager
 * Method:    doUnregister
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_VolumeManager_doUnregister(JNIEnv* env, jobject /*obj*/) {
    if (!context.volumeManagerClass || !context.volumeManagerInstance) {
        ThrowJavaException(env, "java/lang/Exception", "VolumeManager is not registered");
        return;
    }

    YandexIO::getSDKSingleton()->getEndpointStorage()->getLocalEndpoint()->removeCapability(context.volumeCapability);

    context.volumeCapability = nullptr;

    context.volumeManagerNative = nullptr;

    context.volumeManagerClass = NJni::TGlobalClassRef();
    context.volumeManagerInstance = NJni::TGlobalRef();
}

/*
 * Class:     ru_yandex_io_sdk_jni_VolumeManager
 * Method:    doReportVolume
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_VolumeManager_doReportVolume(JNIEnv* env, jobject /*obj*/, jint volume, jboolean isMuted) {
    if (!context.volumeManagerClass || !context.volumeManagerInstance) {
        ThrowJavaException(env, "java/lang/Exception", "VolumeManager is not registered");
        return;
    }
    context.volumeManagerNative->reportVolumeState(volume, isMuted);
}
