#include "jni_capability_wrapper.h"

#include <yandex_io/android_sdk/cpp/jni/wrappers/jni_directive_handler_wrapper.h>
#include <yandex_io/android_sdk/cpp/jni/jni_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace YandexIO {

    JniCapabilityWrapper::JniCapabilityWrapper(JNIEnv* env, jobject instance)
        : capabilityClass_(env->GetObjectClass(instance))
        , capabilityInstance_(instance)
        , getStateMethodId_(env->GetMethodID(capabilityClass_.Get(), "getCapabilityState", "()[B"))
        , getDirectiveHandlerMethodId_(env->GetMethodID(capabilityClass_.Get(),
                                                        "getDirectiveHandler",
                                                        "()Lru/yandex/io/sdk/capability/DirectiveHandler;"))
    {
        init();
    }

    JniCapabilityWrapper::~JniCapabilityWrapper() {
        capabilityClass_ = NJni::TGlobalClassRef();
        capabilityInstance_ = NJni::TGlobalRef();
    }

    const std::string& JniCapabilityWrapper::getId() const {
        return id_;
    }

    NAlice::TCapabilityHolder JniCapabilityWrapper::getState() const {
        auto env = NJni::Env()->GetJniEnv();
        auto jCapability = env->CallObjectMethod(capabilityInstance_.Get(), getStateMethodId_);
        const auto state = static_cast<jbyteArray>(jCapability);

        TInputDataWrapper input(env, state);
        NAlice::TCapabilityHolder capabilityState;
        Y_PROTOBUF_SUPPRESS_NODISCARD capabilityState.ParseFromArray(input.Bytes, input.Size);
        return capabilityState;
    }

    IDirectiveHandlerPtr JniCapabilityWrapper::getDirectiveHandler() {
        auto env = NJni::Env()->GetJniEnv();
        auto jDirectiveHandler = env->CallObjectMethod(capabilityInstance_.Get(), getDirectiveHandlerMethodId_);
        return std::make_shared<JniDirectiveHandlerWrapper>(env, jDirectiveHandler);
    }

    void JniCapabilityWrapper::addListener(std::weak_ptr<IListener> wlistener) {
        listeners_.push_back(wlistener);
    }

    void JniCapabilityWrapper::removeListener(std::weak_ptr<IListener> wlistener) {
        const auto iter = std::find_if(listeners_.begin(), listeners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });
        if (iter != listeners_.end()) {
            listeners_.erase(iter);
        }
    }

    void JniCapabilityWrapper::init() {
        auto env = NJni::Env()->GetJniEnv();

        const jmethodID getId = env->GetMethodID(capabilityClass_.Get(), "getId", "()Ljava/lang/String;");
        auto jId = env->CallObjectMethod(capabilityInstance_.Get(), getId);
        id_ = jstring_to_stdstring(env, static_cast<jstring>(jId));
    }

    void JniCapabilityWrapper::onStateChanged(NAlice::TCapabilityHolder state) {
        for (const auto& wlistener : listeners_) {
            if (auto slistener = wlistener.lock()) {
                slistener->onCapabilityStateChanged(shared_from_this(), state);
            }
        }
    }

} // namespace YandexIO
