#include "jni_directive_handler_wrapper.h"

#include <yandex_io/android_sdk/cpp/jni/jni_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace YandexIO {

    JniDirectiveHandlerWrapper::JniDirectiveHandlerWrapper(JNIEnv* env, jobject instance)
        : directiveHandlerClass_(env->GetObjectClass(instance))
        , directiveHandlerInstance_(instance)
        , handleDirectiveMethodId_(env->GetMethodID(directiveHandlerClass_.Get(), "handleDirective", "([B)V"))
        , cancelDirectiveMethodId_(env->GetMethodID(directiveHandlerClass_.Get(), "cancelDirective", "([B)V"))
        , prefetchDirectiveMethodId_(env->GetMethodID(directiveHandlerClass_.Get(), "prefetchDirective", "([B)V"))
    {
        init();
    }

    JniDirectiveHandlerWrapper::~JniDirectiveHandlerWrapper() {
        directiveHandlerClass_ = NJni::TGlobalClassRef();
        directiveHandlerInstance_ = NJni::TGlobalRef();
    }

    const std::string& JniDirectiveHandlerWrapper::getEndpointId() const {
        return endpointId_;
    }

    const std::string& JniDirectiveHandlerWrapper::getHandlerName() const {
        return handlerName_;
    }

    const std::set<std::string>& JniDirectiveHandlerWrapper::getSupportedDirectiveNames() const {
        return supportedDirectives_;
    }

    void JniDirectiveHandlerWrapper::handleDirective(const std::shared_ptr<Directive>& directive) {
        processDirectiveMethod(handleDirectiveMethodId_, directive);
    }

    void JniDirectiveHandlerWrapper::cancelDirective(const std::shared_ptr<Directive>& directive) {
        processDirectiveMethod(cancelDirectiveMethodId_, directive);
    }

    void JniDirectiveHandlerWrapper::prefetchDirective(const std::shared_ptr<Directive>& directive) {
        processDirectiveMethod(prefetchDirectiveMethodId_, directive);
    }

    void JniDirectiveHandlerWrapper::init() {
        auto env = NJni::Env()->GetJniEnv();

        const jmethodID getEndpointId = env->GetMethodID(directiveHandlerClass_.Get(),
                                                         "getEndpointId", "()Ljava/lang/String;");
        auto jEndpointId = env->CallObjectMethod(directiveHandlerInstance_.Get(), getEndpointId);
        endpointId_ = jstring_to_stdstring(env, static_cast<jstring>(jEndpointId));

        const jmethodID getHandlerName = env->GetMethodID(directiveHandlerClass_.Get(),
                                                          "getHandlerName", "()Ljava/lang/String;");
        auto jHandlerName = env->CallObjectMethod(directiveHandlerInstance_.Get(), getHandlerName);
        handlerName_ = jstring_to_stdstring(env, static_cast<jstring>(jHandlerName));

        const jmethodID getSupportedDirectiveNames = env->GetMethodID(directiveHandlerClass_.Get(),
                                                                      "getSupportedDirectiveNames",
                                                                      "()[Ljava/lang/String;");
        auto jDirectives = env->CallObjectMethod(directiveHandlerInstance_.Get(), getSupportedDirectiveNames);
        const auto jDirectivesArr = static_cast<jobjectArray>(jDirectives);
        const int count = env->GetArrayLength(jDirectivesArr);

        for (int i = 0; i < count; i++) {
            const auto directive = static_cast<jstring>(env->GetObjectArrayElement(jDirectivesArr, i));
            supportedDirectives_.insert(jstring_to_stdstring(env, directive));
        }
    }

    void JniDirectiveHandlerWrapper::processDirectiveMethod(const jmethodID& methodId,
                                                            const std::shared_ptr<Directive>& directive) {
        auto env = NJni::Env();
        auto directiveProtobuf = Directive::convertToDirectiveProtobuf(directive);
        const int size = directiveProtobuf.ByteSize();
        std::vector<char> bytes(size);
        Y_PROTOBUF_SUPPRESS_NODISCARD directiveProtobuf.SerializeToArray(bytes.data(), size);

        auto jBytes = env->NewByteArray(size);
        env->SetByteArrayRegion(jBytes.Get(), 0, size, bytes.data());
        env->CallVoidMethod(directiveHandlerInstance_.Get(), methodId, jBytes.Get());
    }
} // namespace YandexIO
