#include "yandex_iosdk_jni.h"

#include "jni_utils.h"

#include <library/cpp/jni/jni.h>
#include <alice/megamind/protos/speechkit/request.pb.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/base/directive_types.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/vins_utils/vins_utils.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/converters/wifi_info_converter.h>
#include <yandex_io/sdk/proto/device_sdk.pb.h>

YIO_DEFINE_LOG_MODULE("android_sdk");

using namespace quasar;

namespace {

    /**
     * Delegates all calls to attached Java object.
     */
    class AliceRequestEventsJniCallback: public YandexIO::IAliceRequestEvents {
    public:
        AliceRequestEventsJniCallback(JNIEnv* env, jobject instance)
            : aliceRequestCallbackClass_(env->GetObjectClass(instance))
            , aliceRequestCallbackInstance_(instance)
            , onStartMethodID_(env->GetMethodID(aliceRequestCallbackClass_.Get(), "onStarted",
                                                "(Ljava/lang/String;)V"))
            , onSuccessMethodID_(env->GetMethodID(aliceRequestCallbackClass_.Get(), "onSuccess",
                                                  "(Ljava/lang/String;Ljava/lang/String;)V"))
            , onErrorMethodID_(env->GetMethodID(aliceRequestCallbackClass_.Get(), "onError",
                                                "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V"))
        {
        }

        ~AliceRequestEventsJniCallback() = default;

        void onAliceRequestStarted(std::shared_ptr<YandexIO::VinsRequest> request) override {
            auto env = NJni::Env();
            auto jRequestId = env->NewStringUTF(request->getId());
            env->CallVoidMethod(aliceRequestCallbackInstance_.Get(), onStartMethodID_, jRequestId.Get());
        }

        void onAliceRequestCompleted(std::shared_ptr<YandexIO::VinsRequest> request,
                                     const Json::Value& response) override {
            auto env = NJni::Env();
            auto jRequestId = env->NewStringUTF(request->getId());
            auto jResponse = env->NewStringUTF(jsonToString(response, true));
            env->CallVoidMethod(aliceRequestCallbackInstance_.Get(), onSuccessMethodID_, jRequestId.Get(),
                                jResponse.Get());
        }

        void onAliceRequestError(std::shared_ptr<YandexIO::VinsRequest> request, const std::string& errorCode,
                                 const std::string& errorText) override {
            auto env = NJni::Env();
            auto jRequestId = env->NewStringUTF(request->getId());
            auto jErrorCode = env->NewStringUTF(errorCode);
            auto jErrorText = env->NewStringUTF(errorText);
            env->CallVoidMethod(aliceRequestCallbackInstance_.Get(), onErrorMethodID_, jRequestId.Get(),
                                jErrorCode.Get(), jErrorText.Get());
        }

    private:
        NJni::TGlobalClassRef aliceRequestCallbackClass_;
        NJni::TGlobalRef aliceRequestCallbackInstance_;
        jmethodID onStartMethodID_;
        jmethodID onSuccessMethodID_;
        jmethodID onErrorMethodID_;
    };

    std::shared_ptr<AliceRequestEventsJniCallback> getAliceRequestEventsJniCallback(JNIEnv* env, jobject callback) {
        if (!env->IsSameObject(callback, nullptr)) {
            return std::make_shared<AliceRequestEventsJniCallback>(env, callback);
        } else {
            return nullptr;
        }
    }
} // namespace

//
//  Native methods
//

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    toggleConversation
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_toggleConversation(JNIEnv* jenv, jobject /*obj*/, jbyteArray eventSource)
{
    /* Fixme: Find another way to get YandexIOSDK instance. Need to make sure that
     * launcher start was called
     */
    try {
        TInputDataWrapper input(jenv, eventSource);
        NAlice::TSpeechKitRequestProto_TEventSource protobuf;
        Y_ENSURE(protobuf.ParseFromArray(input.Bytes, input.Size));

        YandexIO::getSDKSingleton()->getAliceCapability()->toggleConversation(protobuf);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("YandexIOSDK.JniToggleConversationError", e.what());
    }
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    startConversation
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_startConversation(JNIEnv* jenv, jobject /*obj*/, jbyteArray eventSource)
{
    try {
        TInputDataWrapper input(jenv, eventSource);
        NAlice::TSpeechKitRequestProto_TEventSource protobuf;
        Y_ENSURE(protobuf.ParseFromArray(input.Bytes, input.Size));

        YandexIO::getSDKSingleton()->getAliceCapability()->startConversation(protobuf);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("YandexIOSDK.JniStartConversationError", e.what());
    }
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    stopConversation
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_stopConversation(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getAliceCapability()->stopConversation();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    stopAlarm
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_stopAlarm(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getAlarmCapability()->stopAlarm();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    finishConversationVoiceInput
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_finishConversationVoiceInput(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getAliceCapability()->finishConversationVoiceInput();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeAuthenticate
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeAuthenticate(JNIEnv* env, jobject /*obj*/, jstring xcode)
{
    std::string xcodeStr = jstring_to_stdstring(env, xcode);
    YandexIO::getSDKSingleton()->authenticate(xcodeStr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativePlaySoundFile
 * Signature: (Ljava/lang/String;IZ)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativePlaySoundFile(JNIEnv* env, jobject /*obj*/, jstring fileName, jint channelInt, jboolean playLooped)
{
    std::string fileNameStr = jstring_to_stdstring(env, fileName);
    std::optional<proto::AudioChannel> channel = std::nullopt;
    if (proto::AudioChannel_IsValid(channelInt)) {
        channel = static_cast<proto::AudioChannel>(channelInt);
    }
    YandexIO::IFilePlayerCapability::PlayParams playParams{
        .playLooped = (bool)playLooped};
    YandexIO::getSDKSingleton()->getFilePlayerCapability()->playSoundFile(std::move(fileNameStr), channel, std::move(playParams));
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeStopSoundFile
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeStopSoundFile(JNIEnv* env, jobject /*obj*/, jstring fileName)
{
    std::string fileNameStr = jstring_to_stdstring(env, fileName);
    YandexIO::getSDKSingleton()->getFilePlayerCapability()->stopSoundFile(std::move(fileNameStr));
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeProvideUserAccountInfo
 * Signature: (Ljava/lang/String;Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeProvideUserAccountInfo(JNIEnv* env, jobject /*obj*/, jstring authToken, jstring uid)
{
    std::string authTokenStr = jstring_to_stdstring(env, authToken);
    std::string uidStr = jstring_to_stdstring(env, uid);
    YandexIO::getSDKSingleton()->provideUserAccountInfo(authTokenStr, uidStr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeRegisterDeviceInBackend
 * Signature: (Ljava/lang/String;Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeRegisterDeviceInBackend(JNIEnv* env, jobject /*obj*/, jstring authToken, jstring uid)
{
    std::string authTokenStr = jstring_to_stdstring(env, authToken);
    std::string uidStr = jstring_to_stdstring(env, uid);
    YandexIO::getSDKSingleton()->registerDeviceInBackend(authTokenStr, uidStr);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeRegisterDeviceInBackend
 * Signature: (Ljava/lang/String;Ljava/lang/String;)V
 */
JNIEXPORT jboolean JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeRegisterDeviceInBackendSync(JNIEnv* env, jobject /*obj*/, jstring authToken, jstring uid, jintArray resultCode, jobjectArray message)
{
    std::string authTokenStr = jstring_to_stdstring(env, authToken);
    std::string uidStr = jstring_to_stdstring(env, uid);
    auto status = YandexIO::getSDKSingleton()->registerDeviceInBackendSync(authTokenStr, uidStr);

    env->SetIntArrayRegion(resultCode, 0, 1, &status.responseCode);
    env->SetObjectArrayElement(message, 0, env->NewStringUTF(status.errorMessage.data()));

    return status.isSuccess;
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    revokeUserAccountInfo
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_revokeUserAccountInfo(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->revokeUserAccountInfo();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendTvPolicyInfo
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendTvPolicyInfo(JNIEnv* env, jobject /*obj*/, jbyteArray tvPolicyInfoSerialized)
{
    TInputDataWrapper input(env, tvPolicyInfoSerialized);
    proto::TvPolicyInfo tvPolicyInfoMsg;
    Y_PROTOBUF_SUPPRESS_NODISCARD tvPolicyInfoMsg.ParseFromArray(input.Bytes, input.Size);

    YandexIO::TvPolicyInfo tvPolicyInfo;

    if (tvPolicyInfoMsg.has_content_settings()) {
        tvPolicyInfo.contentSettings = tvPolicyInfoMsg.content_settings();
    }

    if (tvPolicyInfoMsg.has_age_limit()) {
        tvPolicyInfo.ageLimit = tvPolicyInfoMsg.age_limit();
    }

    YandexIO::getSDKSingleton()->reportTvPolicyInfo(tvPolicyInfo);
}

JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendIsScreenActive(JNIEnv* /*jenv*/, jobject /*obj*/, jboolean isScreenActive)
{
    YandexIO::getSDKSingleton()->reportScreenActive((bool)isScreenActive);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    togglePlayPause
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_togglePlayPause(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->togglePlayPause(false);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    nextTrack
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_nextTrack(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->next();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    prevTrack
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_prevTrack(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->prev();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    pause
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_pause(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->pause();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    like
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_like(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->like();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    dislike
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_dislike(JNIEnv* /*jenv*/, jobject /*obj*/)
{
    YandexIO::getSDKSingleton()->getPlaybackControlCapability()->dislike();
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    sendTextQuery
 * Signature: (Ljava/lang/String;Lru/yandex/io/sdk/jni/JniAliceRequestCallback;[B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_sendTextQuery(JNIEnv* jenv, jobject /*obj*/, jstring textQuery, jobject callback, jbyteArray eventSource)
{
    try {
        TInputDataWrapper input(jenv, eventSource);
        NAlice::TSpeechKitRequestProto_TEventSource protobuf;
        Y_ENSURE(protobuf.ParseFromArray(input.Bytes, input.Size));

        auto eventsCallback = getAliceRequestEventsJniCallback(jenv, callback);
        auto event = YandexIO::VinsRequest::buildTextInputEvent(jstring_to_stdstring(jenv, textQuery));
        auto request = std::make_shared<YandexIO::VinsRequest>(std::move(event), std::move(protobuf));

        YandexIO::getSDKSingleton()->getAliceCapability()->startRequest(std::move(request), std::move(eventsCallback));
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("YandexIOSDK.JniSendTextQueryError", e.what());
    }
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    sendUnstructuredServerRequest
 * Signature: (Ljava/lang/String;Z;[B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_sendUnstructuredServerRequest(
    JNIEnv* jenv, jobject /*obj*/, jstring request, jboolean enqueued, jboolean parallel, jobject callback, jbyteArray eventSource)
{
    try {
        TInputDataWrapper input(jenv, eventSource);
        NAlice::TSpeechKitRequestProto_TEventSource protobuf;
        Y_ENSURE(protobuf.ParseFromArray(input.Bytes, input.Size));

        auto eventsCallback = getAliceRequestEventsJniCallback(jenv, callback);
        auto event = parseJson(jstring_to_stdstring(jenv, request));
        auto request = std::make_shared<YandexIO::VinsRequest>(std::move(event), std::move(protobuf));

        request->setEnqueued((bool)enqueued);
        request->setIsParallel((bool)parallel);

        YandexIO::getSDKSingleton()->getAliceCapability()->startRequest(std::move(request), std::move(eventsCallback));
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("YandexIOSDK.JniSendUnstructuredServerRequestError", e.what());
    }
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeProvideState
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeProvideState(JNIEnv* env, jobject /*obj*/, jbyteArray statePartSerialized)
{
    TInputDataWrapper input(env, statePartSerialized);
    yandex_io::proto::TDeviceStatePart statePart;
    Y_PROTOBUF_SUPPRESS_NODISCARD statePart.ParseFromArray(input.Bytes, input.Size);

    YandexIO::getSDKSingleton()->provideState(statePart);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeProvideMediaDeviceIdentifier
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeProvideMediaDeviceIdentifier(JNIEnv* env, jobject /*obj*/, jbyteArray paramsSerialized)
{
    TInputDataWrapper input(env, paramsSerialized);
    NAlice::TClientInfoProto::TMediaDeviceIdentifier identifier;
    Y_PROTOBUF_SUPPRESS_NODISCARD identifier.ParseFromArray(input.Bytes, input.Size);

    YandexIO::getSDKSingleton()->provideMediaDeviceIdentifier(identifier);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendWifiList
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendWifiList(JNIEnv* env, jobject /*obj*/, jbyteArray wifiListSerialized)
{
    TInputDataWrapper input(env, wifiListSerialized);
    proto::WifiList protoWifiList;
    Y_PROTOBUF_SUPPRESS_NODISCARD protoWifiList.ParseFromArray(input.Bytes, input.Size);
    auto wifiList = YandexIO::convertWifiListFromProto(protoWifiList);
    YandexIO::getSDKSingleton()->setWifiList(wifiList);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendTimezone
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendTimezone(JNIEnv* env, jobject /*obj*/, jstring timezoneName, jint offsetSec)
{
    YandexIO::getSDKSingleton()->setTimezone(quasar::jstring_to_stdstring(env, timezoneName), offsetSec);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendLocation
 * Signature: (DDD)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendLocation(JNIEnv* /*jenv*/, jobject /*obj*/, jdouble lat, jdouble lon, jdouble accuracy)
{
    YandexIO::getSDKSingleton()->setLocation(lat, lon, accuracy >= 0 ? std::make_optional(accuracy) : std::nullopt);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendActiveAction
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendActiveActions(JNIEnv* env, jobject /*obj*/, jbyteArray activeActionsSerialized)
{
    TInputDataWrapper input(env, activeActionsSerialized);
    NAlice::TDeviceState::TActiveActions activeActions;
    Y_PROTOBUF_SUPPRESS_NODISCARD activeActions.ParseFromArray(input.Bytes, input.Size);
    YandexIO::getSDKSingleton()->setActiveActions(activeActions);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendActiveActionSemanticFrame
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendActiveActionSemanticFrame(JNIEnv* env, jobject /*obj*/, jstring jsonPayload)
{
    YandexIO::getSDKSingleton()->setActiveActionsSemanticFrames(jsonPayload ? std::make_optional(jstring_to_stdstring(env, jsonPayload)) : std::nullopt);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendAudioRewind
 * Signature: (D;Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendAudioRewind(JNIEnv* env, jobject /*obj*/, jdouble amountMs, jstring rewindType)
{
    Json::Value payload;
    payload["amount_ms"] = amountMs;
    payload["type"] = jstring_to_stdstring(env, rewindType);

    Json::Value directive;
    directive["name"] = Directives::AUDIO_REWIND;
    directive["type"] = DirectiveTypes::LOCAL_ACTION;
    directive["payload"] = payload;

    std::list<Json::Value> directives;
    directives.push_back(directive);
    std::string vinsResponse = createVinsResponsePayload(directives);

    YandexIO::getSDKSingleton()->sendVinsResponse(vinsResponse);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeExecuteDirectiveSequence
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeExecuteDirectiveSequence(JNIEnv* env, jobject /*obj*/, jbyteArray directivesSerialized)
{
    TInputDataWrapper input(env, directivesSerialized);
    proto::DirectiveEntityList protoDirectiveEntityList;
    Y_PROTOBUF_SUPPRESS_NODISCARD protoDirectiveEntityList.ParseFromArray(input.Bytes, input.Size);

    std::list<Json::Value> directives;
    for (auto i = 0; i < protoDirectiveEntityList.directives_size(); ++i) {
        const auto& directiveProto = protoDirectiveEntityList.directives(i);

        Json::Value directive;
        directive["name"] = directiveProto.name();
        if (directiveProto.has_type()) {
            directive["type"] = directiveProto.type();
        }
        directive["payload"] = tryParseJsonOrEmpty(directiveProto.payload());

        directives.push_back(directive);
    }

    std::string vinsResponse = createVinsResponsePayload(directives);
    YandexIO::getSDKSingleton()->sendVinsResponse(vinsResponse);
}

/*
 * Class:     ru_yandex_io_sdk_jni_YandexIOSDK
 * Method:    doNativeSendIsMicrophoneMuted
 * Signature: (Z)V
 */
JNIEXPORT void JNICALL Java_ru_yandex_io_sdk_jni_YandexIOSDK_doNativeSendIsMicrophoneMuted(JNIEnv* /*jenv*/, jobject /*obj*/, jboolean muted)
{
    YandexIO::getSDKSingleton()->getDeviceStateCapability()->setMicsMuted((bool)muted);
}
