#include "generic_first_run_service.h"

#include "global_context.h"

#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/retry_delay_counter.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/jwt/jwt.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/registration_result.h>
#include <yandex_io/services/pushd/xiva_operations.h>

#include <algorithm>

YIO_DEFINE_LOG_MODULE("android_sdk");

using namespace quasar;
using namespace quasar::proto;

namespace {
    const int RETRY_DELAY_INIT_MS = 15000 / 2; // to have 15 sec in average
    const int RETRY_DELAY_MAX_MS = 120000 / 2; // to have 120 sec in average
    const int RETRIES_COUNT = 4;
} // namespace

GenericFirstRunService::GenericFirstRunService(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider)
    : device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , deviceContext_(ipcFactory_, nullptr, false)
    , currentConfigurationState_(ConfigurationState::CONFIGURED)
    , pushdConnector_(ipcFactory_->createIpcConnector("pushd"))
    , authProvider_(std::move(authProvider))
    , backendClient_("quasar-backend", device_)
    , checkToken_(device_, authProvider_, deviceStateProvider)
    , backoffer_(getCrc32(device_->deviceId()))
{
    backoffer_.initCheckPeriod(
        std::chrono::milliseconds(RETRY_DELAY_INIT_MS),
        std::chrono::milliseconds(RETRY_DELAY_INIT_MS),
        std::chrono::milliseconds(RETRY_DELAY_MAX_MS));
}

GenericFirstRunService::~GenericFirstRunService()
{
    lifetime_.die();
    pushdConnector_->shutdown();
}

std::string GenericFirstRunService::getServiceName() const {
    return "firstrund";
}

void GenericFirstRunService::start() {
    backendUrl_ = getString(device_->configuration()->getServiceConfig("common"), "backendUrl");

    const auto config = device_->configuration()->getServiceConfig("firstrund");
    softwareVersion_ = getString(device_->configuration()->getServiceConfig("common"), "softwareVersion");

    backendClient_.setTimeout(std::chrono::milliseconds{9900}); // Borrowed 5*100ms for a pre-sleep, return after SK-4122
    backendClient_.setRetriesCount(5);

    auto cryptographyConfig = getJson(device_->configuration()->getServiceConfig("common"), "cryptography");
    deviceCryptography_ = device_->hal()->createDeviceCryptography(cryptographyConfig);

    server_ = ipcFactory_->createIpcServer(getServiceName());
    server_->setClientConnectedHandler([this](auto& connection) {
        QuasarMessage message;
        message.set_configuration_state(currentConfigurationState_);
        connection.send(std::move(message));
    });
    server_->listenService();

    pushdConnector_->setMessageHandler(std::bind(&GenericFirstRunService::handlePushdMessage, this, std::placeholders::_1));
    pushdConnector_->connectToService();

    deviceContext_.onRegistrationInBackendRequested = makeSafeCallback(
        [this](const std::string& authToken, const std::string& passportUid, OnRegistrationResult onResult) {
            YIO_LOG_DEBUG("YandexIOSDK registration requested")
            onRegistrationRequested(authToken, passportUid, std::move(onResult));
        },
        lifetime_);
    deviceContext_.connectToSDK();
}

void GenericFirstRunService::onRegistrationRequested(const std::string& authToken, const std::string& passportUid,
                                                     OnRegistrationResult onResult) {
    YIO_LOG_DEBUG("Firstrun: has auth info");
    scheduleRegistrationFirstTime(authToken, passportUid, std::move(onResult));
}

void GenericFirstRunService::scheduleRegistrationFirstTime(std::string authToken, std::string uid,
                                                           OnRegistrationResult onResult) {
    std::lock_guard<std::mutex> lock(registrationMutex_);
    YIO_LOG_DEBUG("Firstrun: schedule registration first time");

    registrationLifetime_.die();
    backoffer_.resetDelayBetweenCallsToDefault();

    registrationQueue_.add(
        [this, authToken{std::move(authToken)}, uid{std::move(uid)}, onResult{std::move(onResult)}]() mutable {
            doRegistration(std::move(authToken), std::move(uid), 0, std::move(onResult));
        },
        registrationLifetime_.tracker());
}

void GenericFirstRunService::doRegistration(std::string authToken, std::string uid, int attemptsDone,
                                            OnRegistrationResult onResult) {
    YIO_LOG_DEBUG("Firstrun: attempt to register: " << uid << ", " << maskToken(authToken));
    if (authToken.empty()) {
        YIO_LOG_DEBUG("Firstrun: auth token is empty, skip register");
        YandexIO::getSDKSingleton()->provideUserAccountInfo(authToken, uid);
        onResult(YandexIO::RegistrationResult(false, -1, "Auth token is empty"));
        return;
    }
    BackendResponse response = registerOnBackend(authToken);
    deviceContext_.fireAuthenticationStatus(authToken, response.isSuccess, response.errorMessage);
    if (response.isSuccess) {
        YIO_LOG_DEBUG("Firstrun: successfully registered");
        YandexIO::getSDKSingleton()->provideUserAccountInfo(authToken, uid);
        onResult(YandexIO::RegistrationResult(response.isSuccess, response.code, response.errorMessage));
        return;
    }
    if (response.canRetry && attemptsDone < RETRIES_COUNT) {
        YIO_LOG_WARN("Firstrun: fail to register. Retry");
        rescheduleRegistrationWithDelay(std::move(authToken), std::move(uid), ++attemptsDone, std::move(onResult));
    } else {
        YIO_LOG_ERROR_EVENT("GenericFirstRunService.RegisterFailed", "Firstrun: fail to register");
        onResult(YandexIO::RegistrationResult(response.isSuccess, response.code, response.errorMessage));
    }
}

void GenericFirstRunService::rescheduleRegistrationWithDelay(std::string authToken, std::string uid, int attemptsDone,
                                                             OnRegistrationResult onResult) {
    std::lock_guard<std::mutex> lock(registrationMutex_);

    const auto delay = backoffer_.getDelayBetweenCalls();
    YIO_LOG_DEBUG("Firstrun: reschedule registration with delay " << delay.count() << " ms");
    registrationQueue_.addDelayed(
        [this, authToken{std::move(authToken)}, uid{std::move(uid)}, attemptsDone, onResult{std::move(onResult)}]() mutable {
            doRegistration(std::move(authToken), std::move(uid), attemptsDone, std::move(onResult));
        },
        delay,
        registrationLifetime_.tracker());

    backoffer_.increaseDelayBetweenCalls();
}

GenericFirstRunService::BackendResponse GenericFirstRunService::registerOnBackend(const std::string& oauthToken) {
    const std::string authCode = "0"; // TODO: proper xtoken here
    const std::string urlParams = "device_id=" +
                                  urlEncode(device_->deviceId()) +
                                  "&name=quasar&activation_code=" + std::to_string(getCrc32(authCode)) +
                                  "&firmware_version=" + urlEncode(softwareVersion_) +
                                  "&platform=" + urlEncode(device_->platform()) +
                                  "&ts=" + std::to_string(time(nullptr));
    const std::string backendUrl = backendUrl_ + "/register?" + urlParams;

    YIO_LOG_INFO("Backend request: " << backendUrl);

    HttpClient::Headers headers = {{"Authorization", "OAuth " + oauthToken}};

    try {
        const auto signature = deviceCryptography_->sign(urlParams);
        HttpClient::addSignatureHeaders(headers, signature, deviceCryptography_->getType());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GenericFirstRunService.SignUrlParamsFailed", "Can't sign urlParams (signature version 2): " << e.what());
        // Don't fail here. Try to register without signature
        YIO_LOG_WARN("Trying to register device without signature");
    }

    // Sleep so that passport's database gets a chance to sync newly issued oauth-token, drop this abomination after resolving SK-4122
    std::this_thread::sleep_for(std::chrono::milliseconds(500u));
    HttpClient::HttpResponse response;
    try {
        response = backendClient_.post("register", backendUrl, "{}", headers);
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("GenericFirstRunService.RegisterHttpRequestFailed", "Can't send register request: " << e.what());
        return {.isSuccess = false, .code = -1, .errorMessage = e.what(), .canRetry = true};
    }

    YIO_LOG_INFO("Backend returned code: " << response.responseCode << ". Body: " << response.body);

    if (isSuccessHttpCode(response.responseCode)) {
        return {.isSuccess = true, .code = response.responseCode, .errorMessage = "", .canRetry = true};
    }

    YIO_LOG_ERROR_EVENT("GenericFirstRunService.RegisterNon200Code", "Can't register on backend because wrong response code: " << response.responseCode);

    auto errorMessage = getStringSafe(response.body, ".message");
    if (response.responseCode == 403 && errorMessage == "AUTH_TOKEN_INVALID") {
        YIO_LOG_INFO("GenericFirstRunService register. Fire oauth token is invalid");
        deviceContext_.fireOAuthTokenIsInvalid(oauthToken);
    }
    return {.isSuccess = false, .code = response.responseCode, .errorMessage = errorMessage, .canRetry = canRetryAfterCode(response.responseCode)};
}

bool GenericFirstRunService::canRetryAfterCode(int httpCode) {
    return httpCode >= 500 && httpCode <= 599;
}

void GenericFirstRunService::handlePushdMessage(const quasar::ipc::SharedMessage& message) {
    YIO_LOG_DEBUG("Handle push notification: " << message->push_notification().operation());
    if (message->has_push_notification() && message->push_notification().operation() == XivaOperations::SWITCH_USER) {
        YIO_LOG_DEBUG("switch_user push notification accepted");
        const auto& payload = message->push_notification().message();

        if (payload.empty()) {
            YIO_LOG_ERROR_EVENT("GenericFirstRunService.SwitchUserInvalidPush", "Empty switch_user event payload");
            return;
        }

        Json::Value messageJson;
        try {
            messageJson = parseJson(payload);
        } catch (const Json::Exception& e) {
            YIO_LOG_ERROR_EVENT("GenericFirstRunService.SwitchUserInvalidJson", "Invalid JSON in switch_user payload: " << e.what() << ' ' << payload);
            return;
        }

        if (!messageJson.isMember("token")) {
            YIO_LOG_ERROR_EVENT("GenericFirstRunService.SwitchUserNoJwtToken", "JWT token not found in switch_user payload: " << payload);
            return;
        }

        const std::string jwtToken = messageJson["token"].asString();
        if (checkToken_.check(jwtToken)) {
            YIO_LOG_DEBUG("JWT token from switch_user payload is ok");
            const auto jwt = decodeJWT(jwtToken);
            const std::string xCode = getStringGrantFromJWT(jwt.get(), "x_code");

            auto accountManager = GlobalContext::get().getAccountManager();
            if (accountManager) {
                accountManager->switchUser(xCode);
                YIO_LOG_DEBUG("User switched successfully");
            } else {
                YIO_LOG_ERROR_EVENT("GenericFirstRunService.SwitchUserFailed", "Failed to switch user");
            }
        } else {
            YIO_LOG_ERROR_EVENT("GenericFirstRunService.SwitchUserInvalidJwtToken", "JWT token from switch_user payload is invalid");
            return;
        }
    }
}
