#include "glagol_launcher_service.h"

#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

class GlagolConfigObserver: public YandexIO::BackendConfigObserver {
public:
    GlagolConfigObserver(GlagolLauncherService& launcher)
        : launcher_(launcher)
    {
    }

public:
    void onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) override {
        if (configName != CONFIG_NAME) {
            return;
        }

        const auto config = tryParseJson(jsonConfigValue);
        if (config.has_value()) {
            bool enabled = tryGetBool(*config, "enabled", false);
            if (glagolEnabled_ != enabled) {
                glagolEnabled_ = enabled;
                launcher_.enableGlagolService(glagolEnabled_);
            }
        }
    }

public:
    static const std::string CONFIG_NAME;

private:
    GlagolLauncherService& launcher_;
    bool glagolEnabled_ = false;
};

const std::string GlagolConfigObserver::CONFIG_NAME = "glagold";

GlagolLauncherService::GlagolLauncherService(std::shared_ptr<YandexIO::IDevice> device,
                                             std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                                             std::shared_ptr<IAuthProvider> authProvider,
                                             std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
                                             std::shared_ptr<IMultiroomProvider> multiroomProvider,
                                             std::shared_ptr<IStereoPairProvider> stereoPairProvider,
                                             INsdMessagerFactory& nsdMessagerFactory)
    : device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , authProvider_(std::move(authProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , multiroomProvider_(std::move(multiroomProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , nsdMessagerFactory_(nsdMessagerFactory)
{
    const auto& sdk = YandexIO::getSDKSingleton();
    configObserver_ = std::make_shared<GlagolConfigObserver>(*this);
    sdk->addBackendConfigObserver(configObserver_);
    sdk->subscribeToSystemConfig(GlagolConfigObserver::CONFIG_NAME);
}

GlagolLauncherService::~GlagolLauncherService() {
    glagolService_.reset();
    stubService_.reset();
}

std::string GlagolLauncherService::getServiceName() const {
    return "glagold";
}

void GlagolLauncherService::start() {
    // initially start stub service
    enableGlagolService(false);
}

void GlagolLauncherService::enableGlagolService(bool enabled) {
    std::lock_guard<std::mutex> lock(serviceMutex_);
    YIO_LOG_INFO("enableGlagolService enabled = " << enabled);
    if (glagolService_) {
        glagolService_.reset();
    }
    if (stubService_) {
        stubService_->shutdown();
        stubService_.reset();
    }

    if (!enabled) {
        // start stub
        stubService_ = ipcFactory_->createIpcServer(getServiceName());
        stubService_->setMessageHandler([&](const auto& msg, auto& /* connection */) {
            YIO_LOG_INFO("STUB glagol: Unhandled message for " << getServiceName() << ": " << shortUtf8DebugString(*msg));
        });
        stubService_->listenService();
        YIO_LOG_INFO("glagol stub created");
    } else {
        // start real glagol service
        Glagol::Settings settings;

        // TODO: pass these settings from quasmodrom
        settings.avahi.restrictToIPv4 = true;
        settings.heartbeatTelemetry = false;
        auto glagolConfig = device_->configuration()->getServiceConfig("glagold");
        glagolService_ = std::make_shared<Glagol>(device_, ipcFactory_, authProvider_, deviceStateProvider_,
                                                  multiroomProvider_, stereoPairProvider_,
                                                  createDefaultMndsFactory(nsdMessagerFactory_),
                                                  glagolConfig, settings, YandexIO::getSDKSingleton());
        YandexIO::getSDKSingleton()->getEndpointStorage()->getLocalEndpoint()->addListener(glagolService_);
        YIO_LOG_INFO("glagol real service created");
    }
}
