#include "jni_nsd_messager_bridge.h"

#include <util/system/yassert.h>

using namespace quasar;

void NsdReceiverBridge::onNsdDiscovered(const proto::NsdInfo& nsdInfo) {
    std::scoped_lock lock(mutex_);
    if (receiver_ == nullptr) {
        return;
    }
    receiver_->onNsdDiscovered(nsdInfo);
}

void NsdReceiverBridge::onNsdListDiscovered(const proto::NsdInfoList& nsdInfoList) {
    std::scoped_lock lock(mutex_);
    if (receiver_ == nullptr) {
        return;
    }
    receiver_->onNsdListDiscovered(nsdInfoList);
}

void NsdReceiverBridge::onNsdLost(const proto::NsdInfo& nsdInfo) {
    std::scoped_lock lock(mutex_);
    if (receiver_ == nullptr) {
        return;
    }
    receiver_->onNsdLost(nsdInfo);
}

void NsdReceiverBridge::setGlagolReceiver(INsdReceiver* receiver) {
    Y_VERIFY(receiver != nullptr);
    Y_VERIFY(receiver != this); // deadlock
    std::scoped_lock lock(mutex_);
    receiver_ = receiver;
}

JniNsdMessagerBridge::JniNsdMessagerBridge()
    : guestMode_{false}
    , port_{false}
    , enableNsdSent_{false}
    , jniMessager_{nullptr}
    , mutex_{}
{
}

void JniNsdMessagerBridge::enableNsd(bool guestMode, uint32_t port, OptBool stereopair, OptBool tandem) {
    std::scoped_lock lock(mutex_);
    guestMode_ = guestMode;
    port_ = port;
    stereopair_ = stereopair;
    tandem_ = tandem;
    if (jniMessager_) {
        jniMessager_->enableNsd(guestMode, port, stereopair, tandem);
    }
    enableNsdSent_ = true;
}

void JniNsdMessagerBridge::disableNsd() {
    std::scoped_lock lock(mutex_);
    if (jniMessager_) {
        jniMessager_->disableNsd();
    }
}

void JniNsdMessagerBridge::setJniMessager(INsdMessager* messager) {
    std::scoped_lock lock(mutex_);
    jniMessager_ = messager;
    if (enableNsdSent_) {
        jniMessager_->enableNsd(guestMode_, port_, stereopair_, tandem_);
    }
}

INsdMessagerPtr JniNsdMessagerFactory::createMessager(INsdReceiverPtr receiver) {
    std::scoped_lock lock(mutex_);
    glagolReceiver_ = std::move(receiver);
    auto messager = std::make_shared<JniNsdMessagerBridge>();
    if (jniMessager_) {
        messager->setJniMessager(jniMessager_.get());
        nsdReceiverBridge_.setGlagolReceiver(glagolReceiver_.get());
    }
    glagolMessager_ = messager;
    return messager;
}

INsdReceiver& JniNsdMessagerFactory::nsdReceiver() {
    std::scoped_lock lock(mutex_);
    return nsdReceiverBridge_;
}

void JniNsdMessagerFactory::setJniNsdMessager(INsdMessagerPtr messager) {
    std::scoped_lock lock(mutex_);
    jniMessager_ = std::move(messager);
    if (auto messager = glagolMessager_.lock(); messager && glagolReceiver_) {
        messager->setJniMessager(jniMessager_.get());
        nsdReceiverBridge_.setGlagolReceiver(glagolReceiver_.get());
    }
}
