#include "media_launcher_service.h"

#include <yandex_io/libs/audio_player/factory/player_factory_selector.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/services/mediad/media/players/player_factory.h>

YIO_DEFINE_LOG_MODULE("android_sdk");

MediaLauncherService::MediaLauncherService(std::shared_ptr<YandexIO::IDevice> device,
                                           std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                                           std::shared_ptr<IAuthProvider> authProvider,
                                           std::shared_ptr<IClockTowerProvider> clockTowerProvider,
                                           std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
                                           std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
                                           std::shared_ptr<IMultiroomProvider> multiroomProvider,
                                           std::shared_ptr<IStereoPairProvider> stereoPairProvider,
                                           std::shared_ptr<IUserConfigProvider> userConfigProvider,
                                           std::shared_ptr<YandexIO::SDKInterface> sdk)
    : device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , authProvider_(std::move(authProvider))
    , clockTowerProvider_(std::move(clockTowerProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , glagolClusterProvider_(std::move(glagolClusterProvider))
    , multiroomProvider_(std::move(multiroomProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , userConfigProvider_(std::move(userConfigProvider))
    , sdk_(std::move(sdk))
{
    const auto& netClockConfig = device_->configuration()->getServiceConfig("net_clock");
    if (int localAudioClockPort = tryGetInt(netClockConfig, "port", 0)) {
        try {
            audioClockManager_ = std::make_shared<AudioClockManager>(device_->deviceId(), localAudioClockPort, AudioClockManager::generateClockId());
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("MediaLauncherService.CreateAudioClockManagerFailed", "Fail to start AudioClockManager: " << ex.what());
        }
    }
}

std::string MediaLauncherService::getServiceName() const {
    return "mediad";
}

void MediaLauncherService::start() {
    std::lock_guard<std::mutex> lock(serviceMutex_);

    auto mediadConfig = device_->configuration()->getServiceConfig(getServiceName());
    auto audioPlayerFactory = std::make_shared<PlayerFactorySelector>(mediadConfig);
    auto mediaPlayerFactory = std::make_shared<PlayerFactory>(device_, ipcFactory_, audioClockManager_, audioPlayerFactory, sdk_);
    mediaPlayerFactory->addYandexMusicConstructor(true, multiroomProvider_, stereoPairProvider_, userConfigProvider_);
    mediaPlayerFactory->addYandexRadioConstructor();

    mediaService_ = std::make_unique<MediaService>(device_, ipcFactory_, authProvider_, deviceStateProvider_, multiroomProvider_, stereoPairProvider_, mediaPlayerFactory, sdk_);
    mediaService_->start();
    YIO_LOG_INFO("Media Service started");
}
