#include "quasar_android_launcher.h"

#include "android_hal.h"
#include "generic_first_run_service.h"
#include "glagol_launcher_service.h"
#include "global_context.h"
#include "jni_nsd_messager_bridge.h"
#include "media_launcher_service.h"

#include <yandex_io/android_sdk/cpp/metrics/platform_metrics.h>
#include <yandex_io/android_sdk/cpp/sdk_singleton/sdk_singleton.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/clock_tower/connector/clock_tower_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/glagol/connector/glagol_cluster_provider.h>
#include <yandex_io/interfaces/multiroom/connector/multiroom_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/libs/android_support/system_property/system_property.h>
#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/ipc/mixed/mixed_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/self_destroyer/self_destroyer_utils.h>
#include <yandex_io/libs/speechkit_audio_player/devnull_audio_player.h>
#include <yandex_io/metrica/metrica2d/metrica2_service.h>
#include <yandex_io/metrica/metricad/metrica_service.h>
#include <yandex_io/metrica/metricad/connector/metrica_connector.h>
#include <yandex_io/metrica/monitor/monitor_service.h>
#include <yandex_io/scaffolding/scaffolding.h>
#include <yandex_io/scaffolding/base/endpoint.h>
#include <yandex_io/scaffolding/base/utils.h>
#include <yandex_io/scaffolding/proto/config.pb.h>
#include <yandex_io/sdk/yandex_iosdk.h>
#include <yandex_io/sdk/hub/io_hub_service.h>
#include <yandex_io/services/alarmd/alarm_service.h>
#include <yandex_io/services/aliced/alice_service.h>
#include <yandex_io/services/audiosender/audio_sender_service.h>
#include <yandex_io/services/audiosender/info/info.h>
#include <yandex_io/services/braind/brain_service.h>
#include <yandex_io/services/brickd/brick_service.h>
#include <yandex_io/services/bug_report/bug_report_service.h>
#include <yandex_io/services/fluent-bitd/fluent_bit_service.h>
#include <yandex_io/services/fluent-bitd/flb_lib/fluent_bit_lib.h>
#include <yandex_io/services/networkd/network_service.h>
#include <yandex_io/services/notificationd/notification_service.h>
#include <yandex_io/services/pushd/push_service.h>
#include <yandex_io/services/syncd/sync_service.h>
#include <yandex_io/services/telemetry_proxy/telemetry_proxy_service.h>

#include <yandex_io/modules/xiva_alice_request/xiva_push_alice_requester.h>
#include <yandex_io/modules/spotter_configurer/spotter_configurer.h>

#include <speechkit/Logger.h>

#include <iostream>
#include <memory>

#include <unistd.h>
#include <sys/resource.h>
#include <sys/time.h>

using namespace quasar;

using quasar::proto::QuasarMessage;

namespace {
    const std::string LAUNCHER_SERVICE_NAME = "android_app";

    /* Fake Wifid that will send ACK to requests (make firstrund  happy) */
    class FakeWifiService: public QuasarService {
        std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        std::shared_ptr<ipc::IServer> server_;

    public:
        FakeWifiService(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
            : ipcFactory_(ipcFactory)
        {
        }

        std::string getServiceName() const override {
            return "wifid";
        }

        void start() override {
            server_ = ipcFactory_->createIpcServer(getServiceName());
            server_->setMessageHandler([](const auto& msg, auto& connection) {
                QuasarMessage response;
                response.set_request_id(msg->request_id());
                if (msg->has_wifi_list_request()) {
                    response.mutable_wifi_list();
                } else if (msg->has_wifi_connect()) {
                    auto wifi = response.mutable_wifi_connect_response();
                    wifi->set_status(proto::WifiConnectResponse::OK);
                }
                connection.send(std::move(response));
            });
            server_->listenService();
        }
    };

    std::unique_ptr<YandexIO::ITelemetry> makeTelemetry() {
        return std::make_unique<MetricaConnector>(GlobalContext::get().getIpcFactory(), "metricad");
    }
} // namespace

QuasarAndroidLauncher::QuasarAndroidLauncher(std::shared_ptr<const LauncherConfig> config)
    : config_(config)
{
    /**
     * Init Curl before any thread run. It is necessary because curl_global_init is NOT THREAD SAFE
     */
    quasar::HttpClient::globalInit();
}

QuasarAndroidLauncher::~QuasarAndroidLauncher() {
    GlobalContext::get().setDevice(nullptr);
    GlobalContext::get().setNsdMessagerFactory(nullptr);
}

void QuasarAndroidLauncher::init()
{
    const auto& configPath = config_->paths().config_path();
    YandexIO::ConfigPatterns patterns;
    auto configPlaceholders = config_->paths().config_placeholders();

    for (const auto& pattern : configPlaceholders) {
        patterns[pattern.first] = pattern.second;
    }

    auto configuration = YandexIO::makeConfiguration(configPath, patterns);

    auto ipcFactoryContext = ipc::MixedIpcFactory::createDefaultContext(configuration);
    if (!config_->use_network_ipc()) {
        ipcFactoryContext->transport = ipc::MixedIpcFactory::Transport::LOCAL;
    }

    ipcFactory_ = std::make_shared<ipc::MixedIpcFactory>(std::move(ipcFactoryContext), LAUNCHER_SERVICE_NAME);
    GlobalContext::get().setIpcFactory(ipcFactory_);

    device_ =
        std::make_shared<YandexIO::Device>(
            config_->device_id(),
            configuration,
            makeTelemetry(),
            std::make_unique<AndroidHAL>());
    GlobalContext::get().setDevice(device_);
    GlobalContext::get().setNsdMessagerFactory(std::make_shared<JniNsdMessagerFactory>());

    auto launcherConfig = device_->configuration()->getServiceConfig(LAUNCHER_SERVICE_NAME);
    Logging::initLogging(launcherConfig);
    Logging::addLoggingToTelemetryIfNeeded(launcherConfig, device_->telemetry());
    Logging::addLoggingToIpcIfNeeded(launcherConfig, ipcFactory_);

    static std::once_flag minidump_init_flag;
    std::call_once(minidump_init_flag, [&]() {
        YIO_LOG_INFO("Try to init minidump");
        QuasarMinidump::getInstance().init(LAUNCHER_SERVICE_NAME, launcherConfig, device_->telemetry(), device_->softwareVersion());
        YIO_LOG_INFO("Minidump has been initialized");
    });

    YandexIO::getSDKSingleton()->init(ipcFactory_, "android_sdk", device_->deviceId());

    sendDaemonStartMetric(LAUNCHER_SERVICE_NAME, device_);
}

void QuasarAndroidLauncher::populateServices() {
    const bool isMultiroomSupported = device_->configuration()->isFeatureSupported("multiroom");
    const bool isStereoPairSupported = device_->configuration()->isFeatureSupported("stereo_pair");
    auto authProvider = std::make_shared<AuthProvider>(ipcFactory_);
    auto clockTowerProvider = (isMultiroomSupported ? std::make_shared<ClockTowerProvider>(ipcFactory_) : nullptr);
    auto deviceStateProvider = std::make_shared<DeviceStateProvider>(ipcFactory_);
    auto glagolClusterProvider = std::make_shared<GlagolClusterProvider>(ipcFactory_);
    auto multiroomProvider = (isMultiroomSupported ? std::make_shared<MultiroomProvider>(device_, ipcFactory_) : nullptr);
    auto stereoPairProvider = (isStereoPairSupported ? std::make_shared<StereoPairProvider>(device_, ipcFactory_) : nullptr);
    auto userConfigProvider = std::make_shared<UserConfigProvider>(ipcFactory_);
    auto delayTimingsPolicy = std::make_shared<BackoffRetriesWithRandomPolicy>(getCrc32(device_->deviceId()) + getNowTimestampMs());

    // Media & Multiroom service setup
    auto mediaLauncherService = std::make_unique<MediaLauncherService>(
        device_,
        ipcFactory_,
        authProvider,
        clockTowerProvider,
        deviceStateProvider,
        glagolClusterProvider,
        multiroomProvider,
        stereoPairProvider,
        userConfigProvider,
        YandexIO::getSDKSingleton());

    // Actual services:
    addService(std::move(mediaLauncherService));

    addService<YandexIO::IOHubService>(ipcFactory_);

    addService<AlarmService>(device_, ipcFactory_, authProvider, stereoPairProvider);

    if (AudioSender::configState(device_) == AudioSender::ConfigState::DISABLED) {
        addService<AliceService>(
            device_,
            NAlice::TEndpoint::SpeakerEndpointType, // todo: Client should be able to define EndpointType
            YandexIO::getSDKSingleton(),
            ipcFactory_,
            authProvider,
            clockTowerProvider,
            deviceStateProvider,
            glagolClusterProvider,
            multiroomProvider,
            stereoPairProvider,
            userConfigProvider,
            DevnullAudioPlayer::create(),
            SelfDestroyerUtils::createStub(device_));
    } else {
        addService<AudioSenderService>(device_, ipcFactory_, DevnullAudioPlayer::create(), authProvider);
    }

    addService<BrainService>(device_, ipcFactory_);
    addService<BrickService>(device_, ipcFactory_);
    addService<MetricaService>(device_, ipcFactory_, authProvider);
    addService<Metrica2Service>(device_, ipcFactory_);
    addService<NotificationService>(device_, ipcFactory_, authProvider, stereoPairProvider, userConfigProvider, YandexIO::getSDKSingleton());
    addService<PushService>(device_, ipcFactory_, authProvider, userConfigProvider);
    addService<SyncService>(device_, ipcFactory_, authProvider, deviceStateProvider, delayTimingsPolicy);
    addService<NetworkService>(ipcFactory_);
    addService<BugReportService>(device_, ipcFactory_, userConfigProvider);
    addService<TelemetryProxyService>(device_, ipcFactory_);

    auto collector = createMetricsCollector(device_, ipcFactory_, std::make_shared<PingManager>());
    addService<MonitorService>(device_, ipcFactory_, std::move(collector));

    const auto fluentBitConfig = device_->configuration()->getServiceConfig(FluentBitService::SERVICE_NAME);
    addService<FluentBitService>(device_, ipcFactory_, std::make_unique<FluentBitLib>(device_, fluentBitConfig, ipcFactory_));

    // Generic implementations:
    addService<GenericFirstRunService>(device_, ipcFactory_, authProvider, deviceStateProvider);

    // Fake implementations:
    addService<FakeWifiService>(ipcFactory_);

    addService<GlagolLauncherService>(
        device_,
        ipcFactory_,
        authProvider,
        deviceStateProvider,
        multiroomProvider,
        stereoPairProvider,
        *GlobalContext::get().getNsdMessagerFactory());

    GlobalContext::get().setPingManagerWrapper(
        std::make_shared<PingManagerWrapper>(device_, userConfigProvider));

    // Stubs:
    addStubService("bluetoothd");
    addStubService("updatesd");
    addStubService("interfaced");
    addStubService("videod");
    addStubService("do_not_disturb");
    addStubService("iot");
    addStubService("ntpd");

    // convert xiva push to aliceRequests
    xivaAliceRequester_ = YandexIO::XivaPushAliceRequester::install(*YandexIO::getSDKSingleton());

    spotterConfigurer_ = YandexIO::SpotterConfigurer::install(
        std::make_unique<quasar::NamedCallbackQueue>("SpotterConfigurer"), *YandexIO::getSDKSingleton(), device_);
}

template <typename T, typename... Args>
void QuasarAndroidLauncher::addService(Args&&... args) {
    endpoint_->addService(std::unique_ptr<T>(new T(std::forward<Args>(args)...)));
}

template <typename T>
void QuasarAndroidLauncher::addService(std::unique_ptr<T> service) {
    endpoint_->addService(std::move(service));
}

void QuasarAndroidLauncher::addStubService(const std::string& name) {
    endpoint_->addStubService(ipcFactory_, name);
}

void QuasarAndroidLauncher::start() {
    prepareEnvironment(*config_);

    endpoint_ = std::make_unique<YandexIOEndpoint>(device_, "yiod");
    if (!config_->disable_background_services()) {
        populateServices();
    }

    endpoint_->startAll();
    endpoint_->sync();

    if (config_->has_auth_code()) {
        YIO_LOG_INFO("Authenticating in YandexIOSDK");
        YandexIO::getSDKSingleton()->authenticate(config_->auth_code());
    }
}

void QuasarAndroidLauncher::stop() {
    YIO_LOG_INFO("Stop Quasar Service. Shutdown services");
    endpoint_.reset(nullptr);

    // XXX: destroy speechkit logger to avoid issues with its destruction at
    // static destructor stage
    ::SpeechKit::Logger::setInstance(nullptr);
    YandexIO::resetSDK();
}
