#include "module_platform_collector.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/diskstats/disk_stats_info.h>

#include <fstream>

YIO_DEFINE_LOG_MODULE("yandexmodule_2");

using namespace quasar;

namespace {
    constexpr const char* EMMC_LIFETIME_SYSFS_PATH = "/sys/class/mmc_host/emmc/emmc:0001/life_time";
    constexpr const char* EMMC_PRE_EOL_SYSFS_PATH = "/sys/class/mmc_host/emmc/emmc:0001/pre_eol_info";
} // namespace

ModulePlatformCollector::ModulePlatformCollector(
    std::shared_ptr<YandexIO::IDevice> device,
    const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
    const std::shared_ptr<quasar::PingManager>& pingManager,
    std::chrono::milliseconds period)
    : MetricsCollectorBase(std::move(device), ipcFactory, pingManager, {{"/sys/class/thermal/thermal_zone0/temp", {"SOC", 0.001}}})
{
    executor_ = std::make_unique<quasar::PeriodicExecutor>(
        std::bind(&ModulePlatformCollector::collectSystemMetrics, this),
        period);
}

void ModulePlatformCollector::collectPlatformMetrics(IMetricConsumer& metrics)
{
    DiskStatsInfo diskStatsInfo = DiskStatsInfo::getDiskStatsInfo(disksList_, uptime());
    for (const auto& diskDev : diskStatsInfo.devices) {
        metrics.addDGauge(diskDev.name + "-readPerSecKB", diskDev.readsPerSecKB());
        metrics.addDGauge(diskDev.name + "-writePerSecKB", diskDev.writesPerSecKB());
        metrics.addIGauge(diskDev.name + "-readSectors", diskDev.stats.rdSectors);
        metrics.addIGauge(diskDev.name + "-writeSectors", diskDev.stats.wrSectors);
    }

    if (auto emmcInfo = getEmmcInfo()) {
        metrics.addIGauge("emmcLifetimeA", emmcInfo->lifetimeTypeA);
        metrics.addIGauge("emmcLifetimeB", emmcInfo->lifetimeTypeB);
        metrics.addIGauge("emmcPreEOL", emmcInfo->preEOLinfo);
    }
}

std::optional<ModulePlatformCollector::EmmcStatInfo> ModulePlatformCollector::getEmmcInfo() {
    EmmcStatInfo emmcInfo;

    std::ifstream emmcLifetime(EMMC_LIFETIME_SYSFS_PATH);
    if (emmcLifetime.good()) {
        emmcLifetime >> std::hex >> emmcInfo.lifetimeTypeA;
        emmcLifetime >> std::hex >> emmcInfo.lifetimeTypeB;
    } else {
        YIO_LOG_ERROR_EVENT("MetricsCollectorImpl.ReadEmmcLifetimeFailed", "Failed to read sysfs node: " << EMMC_LIFETIME_SYSFS_PATH);
        return std::nullopt;
    }

    std::ifstream emmcPreEOL(EMMC_PRE_EOL_SYSFS_PATH);
    if (emmcPreEOL.good()) {
        emmcPreEOL >> emmcInfo.preEOLinfo;
    } else {
        YIO_LOG_ERROR_EVENT("MetricsCollectorImpl.ReadEmmcPreEolFailed", "Failed to read sysfs node: " << EMMC_PRE_EOL_SYSFS_PATH);
        return std::nullopt;
    }

    return emmcInfo;
}
