#include "ping_manager_wrapper.h"

#include "yandex_io/libs/configuration/configuration.h"
#include "yandex_io/libs/logging/logging.h"
#include "yandex_io/libs/json_utils/json_utils.h"
#include "yandex_io/libs/telemetry/telemetry.h"

#include "yandex_io/android_sdk/cpp/metrics/cold_start_detector/cold_start_detector.h"

#include <chrono>
#include <sys/sysinfo.h>

using namespace quasar;

namespace {
    constexpr auto DEFAULT_QUICK_PING_TIMEOUT = std::chrono::minutes(1);
} // namespace

PingManagerWrapper::PingManagerWrapper(std::shared_ptr<YandexIO::IDevice> device,
                                       std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : pingManager_(new PingManager(std::bind(&PingManagerWrapper::onPingerEvent, this, std::placeholders::_1)))
    , device_(std::move(device))
    , userConfigProvider_(std::move(userConfigProvider))
    , worker_(std::make_shared<NamedCallbackQueue>("PingManagerWrapper"))
    , uniqueCallback_(worker_, UniqueCallback::ReplaceType::REPLACE_FIRST)
{
    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM,
                                           "quick_pinger")
        .connect([this](const auto& json) {
            worker_->add([this, json]() {
                config_ = *json;
            }, lifetime_);

            /* Start quick pings on each device boot. We don't want to read
             * bootId from sdcard (the way ColdStartDetector works) on every
             * config update, so we store configWasReceived_ in memory.
             */
            if (!configWasReceived_) {
                ColdStartDetector coldStartDetector(device_);
                if (coldStartDetector.checkIfColdStart()) {
                    startQuickPings("cold_start");
                }
                configWasReceived_ = true;
            }
        }, lifetime_);
}

void PingManagerWrapper::startQuickPings(const std::string& reason) {
    worker_->add([this, reason]() {
        if (!uniqueCallback_.isScheduled()) { // We don't need several quick ping sessions at once
            YIO_LOG_DEBUG("start quick pings " << jsonToString(config_));
            startedAt_ = uptime();
            pingManager_->start(config_);
            reason_ = reason;

            Json::Value eventPayload;
            eventPayload["startUptime"] = startedAt_;
            eventPayload["reason"] = reason_;
            device_->telemetry()->reportEvent("quickPingStart", jsonToString(eventPayload));

            auto timeout = tryGetMillis(config_, "timeoutMs",
                                        std::chrono::duration_cast<std::chrono::milliseconds>(
                                            DEFAULT_QUICK_PING_TIMEOUT));

            uniqueCallback_.executeDelayed([this, timeout]() {
                pingManager_->stop();
                Json::Value eventPayload;
                eventPayload["startUptime"] = startedAt_;
                eventPayload["timeoutMs"] = static_cast<int64_t>(timeout.count());
                eventPayload["reason"] = reason_;
                device_->telemetry()->reportEvent("quickPingTimeout");
            }, timeout, lifetime_);
        }
    });
}

PingManagerWrapper::~PingManagerWrapper() {
    uniqueCallback_.reset();
    lifetime_.die();
}

void PingManagerWrapper::onPingerEvent(const Pinger::Event& event) {
    if (event.type == Pinger::EventType::PACKET_RECEIVED) {
        worker_->add([this, event]() {
            Json::Value eventPayload;
            eventPayload["host"] = event.host;

            /* We need to know the time elapsed since device start
             * and till the moment of sending the first successful ping request */
            auto receivedUptimeSec = uptime();
            auto sentUptimeSec = receivedUptimeSec - static_cast<int64_t>(std::chrono::duration_cast<std::chrono::seconds>(
                                                                              event.elapsed)
                                                                              .count());

            eventPayload["startUptime"] = startedAt_;
            eventPayload["receivedUptime"] = receivedUptimeSec;
            eventPayload["sentUptime"] = sentUptimeSec;
            eventPayload["reason"] = reason_;

            device_->telemetry()->reportEvent("quickPingReceived",
                                              jsonToString(eventPayload));
            YIO_LOG_DEBUG("quick ping received: " << jsonToString(eventPayload));
            pingManager_->stop();
            uniqueCallback_.reset();
            YIO_LOG_DEBUG("ping manager stopped");
        });
    }
}

void PingManagerWrapper::setNetworkStatus(YandexIO::ITelemetry::IParams::NetworkStatus /*status*/) {
    pingManager_->reloadGateway();
}

int64_t PingManagerWrapper::uptime() {
    struct sysinfo info;
    sysinfo(&info);

    return static_cast<int64_t>(info.uptime);
}
