#include "android_volume_manager.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/modules/volume_manager/base/volume_manager.h>

AndroidVolumeManager::AndroidVolumeManager(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    GetVolumeCallback getVolumeCallback,
    SetVolumeCallback setVolumeCallback,
    AdjustVolumeCallback adjustVolumeCallback,
    IsMutedCallback isMutedCallback,
    SetMutedCallback muteCallback,
    SetMutedCallback unmuteCallback)
    : sdk_(std::move(sdk))
    , volumeMetrics_(device->telemetry())
    , getVolumeCallback_(std::move(getVolumeCallback))
    , setVolumeCallback_(std::move(setVolumeCallback))
    , adjustVolumeCallback_(std::move(adjustVolumeCallback))
    , isMutedCallback_(std::move(isMutedCallback))
    , muteCallback_(std::move(muteCallback))
    , unmuteCallback_(std::move(unmuteCallback))
{
    const auto config = device->configuration()->getServiceConfig("maind");
    cfgAndroidVolume_.min = quasar::tryGetInt(config, "minAndroidVolume", DEFAULT_ANDROID_VOLUME.min);
    cfgAndroidVolume_.max = quasar::tryGetInt(config, "maxAndroidVolume", DEFAULT_ANDROID_VOLUME.max);
    cfgAndroidVolume_.step = quasar::tryGetInt(config, "androidVolumeStep", DEFAULT_ANDROID_VOLUME.step);
    androidVolume_ = cfgAndroidVolume_;
}

AndroidVolumeManager::~AndroidVolumeManager() {
}

void AndroidVolumeManager::start() {
    sdk_->addBackendConfigObserver(shared_from_this());
    reportVolumeState(getVolumeCallback_(), isMutedCallback_());
}

void AndroidVolumeManager::reportVolumeState(int volume, bool isMuted) {
    volumeMetrics_.onVolumeChange(volume, volume, isMuted, VolumeManager::SOURCE_OTHER, false);
    sdk_->getDeviceStateCapability()->setVolumeState(isMuted, volume, androidVolume_.max);
}

void AndroidVolumeManager::volumeUp(const std::string& /*source*/) {
    YIO_LOG_DEBUG("Volume up");
    adjustVolumeCallback_(AdjustVolumeDirection::RAISE);
}

void AndroidVolumeManager::volumeDown(const std::string& /*source*/) {
    YIO_LOG_DEBUG("Volume down");
    adjustVolumeCallback_(AdjustVolumeDirection::LOWER);
}

void AndroidVolumeManager::setVolume(int aliceVolume, bool /*animate*/, const std::string& /*source*/) {
    YIO_LOG_DEBUG("Set volume: " << aliceVolume);
    setVolumeCallback_(aliceVolume);
}

void AndroidVolumeManager::mute() {
    YIO_LOG_DEBUG("Mute");
    muteCallback_();
}

void AndroidVolumeManager::unmute() {
    YIO_LOG_DEBUG("Unmute");
    unmuteCallback_();
}

void AndroidVolumeManager::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != "maind") {
        return;
    }
    const auto config = quasar::tryParseJson(jsonConfigValue);
    if (config.has_value()) {
        handleConfigUpdate(*config);
    }
}

void AndroidVolumeManager::handleConfigUpdate(const Json::Value& config) {
    androidVolume_.min = quasar::tryGetInt(config, "minAndroidVolume", cfgAndroidVolume_.min);
    androidVolume_.max = quasar::tryGetInt(config, "maxAndroidVolume", cfgAndroidVolume_.max);
    androidVolume_.step = quasar::tryGetInt(config, "androidVolumeStep", cfgAndroidVolume_.step);
    YIO_LOG_INFO("Update Android volume: min=" << androidVolume_.min << ", max=" << androidVolume_.max << ", step=" << androidVolume_.step);
}
