#pragma once

#include <yandex_io/libs/device/device.h>
#include <yandex_io/modules/volume_manager/base/volume_metrics.h>
#include <yandex_io/sdk/alice_volume_setter.h>
#include <yandex_io/sdk/backend_config_observer.h>
#include <yandex_io/sdk/sdk_interface.h>

#include <json/json.h>

#include <functional>
#include <mutex>

class AndroidVolumeManager
    : public std::enable_shared_from_this<AndroidVolumeManager>,
      public YandexIO::AliceVolumeSetter,
      public YandexIO::BackendConfigObserver {
public:
    enum class AdjustVolumeDirection {
        LOWER = -1,
        RAISE = 1
    };

    using GetVolumeCallback = std::function<int()>;
    using SetVolumeCallback = std::function<void(int)>;
    using AdjustVolumeCallback = std::function<void(AdjustVolumeDirection)>;
    using IsMutedCallback = std::function<bool()>;
    using SetMutedCallback = std::function<void()>;

    AndroidVolumeManager(
        std::shared_ptr<YandexIO::IDevice> device,
        std::shared_ptr<YandexIO::SDKInterface> sdk,
        GetVolumeCallback getVolumeCallback,
        SetVolumeCallback setVolumeCallback,
        AdjustVolumeCallback adjustVolumeCallback,
        IsMutedCallback isMutedCallback,
        SetMutedCallback muteCallback,
        SetMutedCallback unmuteCallback);

    ~AndroidVolumeManager();

    void start();
    void reportVolumeState(int volume, bool isMuted);

public: // AliceVolumeSetter
    void volumeUp(const std::string& source = YandexIO::AliceVolumeSetter::SOURCE_VOICE) override;
    void volumeDown(const std::string& source = YandexIO::AliceVolumeSetter::SOURCE_VOICE) override;
    void setVolume(int aliceVolume, bool animate = true, const std::string& source = YandexIO::AliceVolumeSetter::SOURCE_VOICE) override;
    void mute() override;
    void unmute() override;

public: // BackendConfigObserver
    void onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) override;

private:
    void handleConfigUpdate(const Json::Value& config);

    const std::shared_ptr<YandexIO::SDKInterface> sdk_;

    VolumeMetrics volumeMetrics_;

    GetVolumeCallback getVolumeCallback_;
    SetVolumeCallback setVolumeCallback_;
    AdjustVolumeCallback adjustVolumeCallback_;
    IsMutedCallback isMutedCallback_;
    SetMutedCallback muteCallback_;
    SetMutedCallback unmuteCallback_;

    struct VolumeRange {
        int min;
        int max;
        int step;
    };

    static constexpr VolumeRange DEFAULT_ANDROID_VOLUME{.min = 0, .max = 100, .step = 5};

    // Initial values from quasar.cfg
    VolumeRange cfgAndroidVolume_ = DEFAULT_ANDROID_VOLUME;

    // These values come from system_config or from cfgAndroidVolumeRange_ if config is empty
    VolumeRange androidVolume_ = DEFAULT_ANDROID_VOLUME;
};
