@file:Suppress("unused")

package com.yandex.tv.yamake

import com.android.build.gradle.AppExtension
import com.android.build.gradle.LibraryExtension
import com.android.build.gradle.internal.core.InternalBaseVariant
import com.android.build.gradle.internal.tasks.factory.dependsOn
import org.gradle.api.Action
import org.gradle.api.DomainObjectSet
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.UnknownTaskException
import org.gradle.api.tasks.Delete
import org.gradle.api.tasks.Exec
import org.slf4j.LoggerFactory
import ru.yandex.external_builder.DISABLE_EXTERNAL_BUILDER_FLAG
import ru.yandex.external_builder.EXTERNAL_BUILD_DIR
import java.io.File
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardCopyOption.COPY_ATTRIBUTES
import java.nio.file.StandardCopyOption.REPLACE_EXISTING
import java.util.Locale

/**
 * Plugin supports integration of ya make builds into gradle builds.
 * Only single jars are supported for now.
 */
open class YaMakePlugin : Plugin<Project> {

    private val logger = LoggerFactory.getLogger("YaMakePlugin")

    override fun apply(project: Project) {
        val extension = project.extensions.create(
            YaMakeExtension.EXT_NAME,
            YaMakeExtension::class.java)

        getAndroidBuildVariants(project).configureEach { variant ->
            addTasksForVariant(project, extension, variant)
        }

        val cleanExternal = project.tasks.register("cleanExternalBuild", Delete::class.java) {
            it.delete(
                getProjectYaBuildDir(project, getArcadiaRootDir(extension)),  // intermediates
                joinPath("${project.projectDir}", EXTERNAL_BUILD_DIR)  // outputs
            )
        }
        project.tasks.named("clean").dependsOn(cleanExternal)
    }

    private fun addTasksForVariant(
        project: Project,
        extension: YaMakeExtension,
        variant: InternalBaseVariant
    ) {
        if (!project.hasProperty(DISABLE_EXTERNAL_BUILDER_FLAG)) {
            configureTaskDependency(project, extension, variant)
        } else {
            configureFileDependency(project, extension, variant)
        }
    }

    private fun getAndroidBuildVariants(project: Project): DomainObjectSet<out InternalBaseVariant> {
        return when (val android = project.properties["android"]) {
            is LibraryExtension -> android.libraryVariants
            is AppExtension -> android.applicationVariants
            else -> throw IllegalArgumentException(
                "Expected project.android to be either " +
                        "com.android.build.gradle.AppExtension or com.android.build.gradle.LibraryExtension, " +
                        "but got ${android?.javaClass?.name}"
            )
        }
    }

    @Suppress("ObjectLiteralToLambda")
    private fun configureTaskDependency(
        project: Project,
        ext: YaMakeExtension,
        buildVariant: InternalBaseVariant
    ) {

        // Single task for different build types i.e. for debug and release.
        // Use flavors to create and configure different tasks.
        val taskName = "yaMake${capitalize(buildVariant.flavorName)}"
        try {
            project.tasks.named(taskName)
            return // this flavor already configured (with another build type)
        } catch (expected: UnknownTaskException) { }

        val yaTool = ext.yaTool.get().asFile.canonicalPath
        val targetDir = ext.targetDir.get().asFile.canonicalPath
        val targetName = ext.targetName.get()
        val arcadiaRoot = getArcadiaRootDir(ext)

        val yaBuildDir = getProjectYaBuildDir(project, arcadiaRoot)
        val intermediateDir = joinPath(yaBuildDir, buildVariant.flavorName)

        val runOnCI = System.getenv("BUILD_NUMBER") != null
        val extraMakeArgs = if (runOnCI) listOf("-G") else listOf()  // Dump full build graph to stdout

        val cmd = arrayListOf(
            yaTool, "make",
            "--build", "release",
            "-o", intermediateDir,
            "--no-src-links",       // Do not create any symlink in source directory
        ) + extraMakeArgs + ext.getYtArgs() + targetDir
        logger.info("yaMake cmd: " + cmd.joinToString(" "))

        val projectRelPath = getProjectRelPath(project, arcadiaRoot)
        val intermediateFile = joinPath(intermediateDir, "$projectRelPath", targetName)
        logger.info("yaMake out: $intermediateFile")

        val outputFile = getOutputFile(project, buildVariant, targetName)

        val task = project.tasks.register(taskName, Exec::class.java, object: Action<Exec> {
            override fun execute(exec: Exec) {
                exec.commandLine(cmd)
                if (runOnCI) { exec.standardOutput = createYaMakeOutputFile(project).outputStream() }
                exec.doLast(object: Action<Task> {
                    override fun execute(t: Task) {
                        // for some reason (vfs bug?) we may or may not see file immediately
                        val outFile = Paths.get(intermediateFile)
                        for (i in 0 until 10) {
                            if (Files.notExists(outFile)) {
                                println("File $outFile not found. wait 1s and re-check")
                                Thread.sleep(1000)
                            } else {
                                break
                            }
                        }
                        // outputFile will have modification time from original file (under ~/.ya/build/...)
                        // modification time of intermediateSymlink does not matter
                        Files.copy(Paths.get(intermediateFile), Paths.get(outputFile),
                            COPY_ATTRIBUTES, REPLACE_EXISTING)
                    }
                })
                exec.outputs.files(outputFile)
                // Run ya make each time (ya make is incremental).
                // BTW if file modification time of outputFile does not change
                // gradle will not invalidate results of dependent tasks.
                // Comment out to save few seconds but lose tracking of changes.
                exec.outputs.upToDateWhen { false }
            }
        })
        val dependencyNotation = project.fileTree(outputFile).builtBy(task)
        project.dependencies.add(apiFor(buildVariant.flavorName), dependencyNotation)
    }

    private fun configureFileDependency(
        project: Project,
        ext: YaMakeExtension,
        buildVariant: InternalBaseVariant
    ) {
        val outputFile = getOutputFile(project, buildVariant, ext.targetName.get())
        val dependencyNotation = project.files(outputFile)
        project.dependencies.add(apiFor(buildVariant.name), dependencyNotation)
    }

    private fun apiFor(configuration: String): String {
        return when {
            configuration.isEmpty() -> "api"
            else -> "${configuration}Api"
        }
    }

    private fun getArcadiaRootDir(ext: YaMakeExtension): String {
        val yaTool = ext.yaTool.get().asFile.canonicalPath
        return File(yaTool).parent!!
    }

    private fun getProjectRelPath(project: Project, arcadiaRoot: String): Path {
        return Paths.get(arcadiaRoot).relativize(Paths.get(project.projectDir.canonicalPath))
    }

    private fun getProjectYaBuildDir(project: Project, arcadiaRoot: String): String {
        val projectRelPath = getProjectRelPath(project, arcadiaRoot)
        return joinPath("$arcadiaRoot/../ya-build", "$projectRelPath")
    }

    private fun getOutputFile(project: Project, buildVariant: InternalBaseVariant, fileName: String): String {
        return joinPath("${project.projectDir}", EXTERNAL_BUILD_DIR, buildVariant.flavorName, fileName)
    }

    private fun createYaMakeOutputFile(project: Project): File {
        val yaMakeOutputDir = File("${project.buildDir}/yaMakeOutput/")
        yaMakeOutputDir.mkdirs()
        return File.createTempFile("make-", ".txt", yaMakeOutputDir)
    }

    private fun joinPath(vararg segments: String): String {
        val pathStr = segments.filter { it.isNotEmpty() }.joinToString(File.separator)
        return File(pathStr).canonicalPath
    }

    private fun capitalize(str: String): String {
        return str.replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.US) else it.toString() }
    }
}
