package ru.yandex.external_builder

import com.android.build.gradle.internal.tasks.MergeNativeLibsTask
import com.android.build.gradle.tasks.MergeSourceSetFolders
import org.gradle.api.DefaultTask
import org.gradle.api.Task
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.tasks.InputDirectory
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity
import org.gradle.api.tasks.TaskAction
import org.gradle.work.Incremental
import org.gradle.work.InputChanges
import org.slf4j.LoggerFactory
import java.io.File
import java.util.Locale

/**
 * ExternalBuilderIncrementalTask is responsible for tracking changes in external build
 * and triggering rerun of subsequent build tasks (such as :mergeDebugAssets or :packageRelease).
 *
 * Entry point to this task is [doTaskAction] method, which accepts [InputChanges] as only parameter.
 *
 * This task operates as follows:
 * 1. This task obtains a list of all files where changes are detected.
 * 2. For each file in list, this task removes a copy of that file from build cache in build/intermediates.
 * 3. Removal of that file from cache will trigger appropriate incremental merge task to rerun.
 * 4. Merge task copies fresh version of that file to build cache.
 * 6. Merge task marks its output as dirty and triggers packaging task to rerun.
 * 7. Packaging task assembles new APK with fresh version of changed file.
 *
 */
abstract class ExternalBuilderIncrementalTask : DefaultTask() {
    @get:Incremental
    @get:InputDirectory
    @get:PathSensitive(PathSensitivity.RELATIVE)
    abstract val inputDir: DirectoryProperty

    /**
     * Output dir is the same as input dir, because this task does not change files
     */
    @get:OutputDirectory
    abstract val outputDir: DirectoryProperty

    @get:Internal
    internal lateinit var variantName: String

    private val logger by lazy { LoggerFactory.getLogger(name) }

    private val buildVariantCapitalized by lazy { variantName.capitalize(Locale.US) }

    @TaskAction
    fun doTaskAction(inputChanges: InputChanges) {
        val mergeTasks = SOURCE_SET_TYPES
            .mapNotNull { findMergeTaskInProject(it) as MergeSourceSetFolders? }
            .map { MergeSourceSetFoldersAdapter(it) }
            .toMutableList<MergeSourcesTask>()

        val mergeNativeLibsTask = findMergeTaskInProject(SOURCE_TYPE_NATIVE_LIBS) as MergeNativeLibsTask?
        if (mergeNativeLibsTask != null) {
            mergeTasks.add(MergeNativeLibsAdapter(mergeNativeLibsTask))
        }

        logger.debug("isIncremental = ${inputChanges.isIncremental}")
        inputChanges.getFileChanges(inputDir).forEach {
            logger.debug("Found change: ${it.changeType} ${it.file}")
            processChange(it.file, mergeTasks)
        }
    }

    private fun processChange(changedFile: File, mergeTasks: List<MergeSourcesTask>) {
        for (task in mergeTasks) {
            for (inputPath in task.getInputDirs().map { it.canonicalPath }) {
                if (changedFile.canonicalPath.startsWith(inputPath)) {
                    val file = File(task.getOutputDir(), changedFile.canonicalPath.removePrefix(inputPath))
                    tryDeleteMergedFile(file)
                }
            }
        }
    }

    private fun tryDeleteMergedFile(merged: File) {
        if (!merged.isFile) {
            return
        }
        if (merged.delete()) {
            logger.info("Merged source deleted: $merged")
        } else {
            logger.error("Cannot delete merged source: $merged")
        }
    }

    private fun findMergeTaskInProject(sourceSetType: String): Task? {
        val taskName = "merge$buildVariantCapitalized$sourceSetType"
        return project.getTasksByName(taskName, /* recursive = */ false).firstOrNull()
    }

    @Internal
    internal fun getDependentTasks(): List<String> {
        return SOURCE_SET_TYPES.map { "merge$buildVariantCapitalized$it" } + listOf(
            "generate${buildVariantCapitalized}Assets",
            "merge$buildVariantCapitalized$SOURCE_TYPE_NATIVE_LIBS",
            "package$buildVariantCapitalized",
            "javaPreCompile$buildVariantCapitalized",
            "assemble$buildVariantCapitalized"
        )
    }

    /**
     * Adapter interface which unifies public interfaces of different merge tasks.
     */
    private interface MergeSourcesTask {
        fun getInputDirs(): Iterable<File>
        fun getOutputDir(): File
    }

    /**
     * Adapter for [MergeSourceSetFolders] task.
     */
    private class MergeSourceSetFoldersAdapter(private val task: MergeSourceSetFolders) : MergeSourcesTask {
        override fun getInputDirs(): MutableSet<File> = task.sourceFolderInputs.files
        override fun getOutputDir(): File = task.outputDir.asFile.get()
    }

    /**
     * Adapter for [MergeNativeLibsTask] task.
     */
    private class MergeNativeLibsAdapter(private val task: MergeNativeLibsTask) : MergeSourcesTask {
        override fun getInputDirs(): Iterable<File> {
            return task.projectNativeLibs.files +
                    task.externalLibNativeLibs.files +
                    task.subProjectNativeLibs.files
        }

        override fun getOutputDir(): File = task.outputDir.asFile.get()
    }

    companion object {
        private const val SOURCE_TYPE_NATIVE_LIBS = "NativeLibs"
        private val SOURCE_SET_TYPES = listOf(
            "Assets",
            "JniLibFolders",
            "MlModels",
            "Shaders"
        )
    }
}
