package ru.yandex.external_builder

import com.android.build.api.dsl.AndroidSourceSet
import com.android.build.gradle.AppExtension
import com.android.build.gradle.LibraryExtension
import com.android.build.gradle.internal.core.InternalBaseVariant
import com.android.build.gradle.internal.tasks.factory.dependsOn
import com.android.builder.model.ProductFlavor
import org.gradle.api.DomainObjectSet
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.tasks.Delete
import org.slf4j.LoggerFactory
import java.io.File
import java.util.Locale

/**
 * [ExternalBuilderPlugin] is the main entry point to external builder.
 *
 * It creates all external builder tasks and embeds them into the task graph of the project.
 *
 * Also it creates [ExternalBuilderExtension] to configure plugin and connects it to Gradle project.
 */
@Suppress("unused")
open class ExternalBuilderPlugin : Plugin<Project> {

    private val logger by lazy { LoggerFactory.getLogger("ExternalBuilderPlugin") }

    override fun apply(project: Project) {
        val flavors = project.container(ExternalBuilderExtension.Flavor::class.java)
        val extension = project.extensions.create(
            ExternalBuilderExtension.EXT_NAME,
            ExternalBuilderExtension::class.java,
            flavors
        )

        project.afterEvaluate {
            project.getAndroidBuildVariants().configureEach { variant ->
                addDependenciesForVariant(project, extension, variant)
            }

            val cleanExternal = project.tasks.register("cleanExternalBuild", Delete::class.java) {
                it.delete("${project.projectDir}/$EXTERNAL_BUILD_DIR")
            }
            project.tasks.named("clean").dependsOn(cleanExternal)
        }
    }

    @Suppress("UnstableApiUsage")
    private fun addDependenciesForVariant(
        project: Project,
        extension: ExternalBuilderExtension,
        variant: InternalBaseVariant
    ) {
        val taskVariant = variant.flavorName
        val outputDir = File("${project.projectDir}/$EXTERNAL_BUILD_DIR/$taskVariant")
        val externalBuildTask = project.createExtBuilderTask(extension, taskVariant, outputDir, variant.productFlavors)
        val incrementalTask = project.createExtBuilderIncrementalTask(variant.name, outputDir)

        if (!project.hasProperty(DISABLE_EXTERNAL_BUILDER_FLAG)) {
            logger.debug("external_builder: ${incrementalTask.name} dependsOn ${externalBuildTask.name}")
            incrementalTask.dependsOn(externalBuildTask)
            for (taskName in incrementalTask.getDependentTasks()) {
                val task = project.tasks.findByName(taskName)
                logger.debug("external_builder: ${task?.name} dependsOn ${incrementalTask.name}")
                task?.dependsOn(incrementalTask)
            }
        } else {
            logger.debug("external_builder: plugin disabled, no tasks created")
        }

        val sourceSet = variant.sourceSets.find { it.name == variant.name }
        sourceSet!!.apply {
            this as AndroidSourceSet
            assets.srcDir(File(outputDir, "assets"))
            jniLibs.srcDir(File(outputDir, "jniLibs/lib"))
        }
    }

    private fun Project.getAndroidBuildVariants(): DomainObjectSet<out InternalBaseVariant> {
        return when (val android = properties["android"]) {
            is LibraryExtension -> android.libraryVariants
            is AppExtension -> android.applicationVariants
            else -> throw IllegalArgumentException(
                "Expected project.android to be either " +
                        "com.android.build.gradle.AppExtension or com.android.build.gradle.LibraryExtension, " +
                        "but got ${android?.javaClass?.name}"
            )
        }
    }

    private fun Project.createExtBuilderTask(
        ext: ExternalBuilderExtension,
        taskVariantName: String,
        outputDir: File,
        flavors: List<ProductFlavor>,
    ): ExternalBuilderTask {
        val taskName = getExternalBuildTaskName(taskVariantName)
        val existingTask = tasks.findByName(taskName)

        if (existingTask != null && existingTask is ExternalBuilderTask) {
            return existingTask
        }

        return tasks.create(taskName, ExternalBuilderTask::class.java) { t ->
            t.packageJson.set(ext.packageJson)
            t.externalFlavors.set(ext.flavors)
            t.outputDir.set(outputDir)
            t.yaTool.set(ext.yaTool)
            t.ytCacheArgs.set(ext.getYtArgsProvider())
            t.flavors = flavors
        }
    }

    private fun Project.createExtBuilderIncrementalTask(
        taskVariantName: String,
        outputDir: File,
    ): ExternalBuilderIncrementalTask {
        val taskName = getExternalBuildIncrementalTaskName(taskVariantName)
        val existingTask = tasks.findByName(taskName)

        if (existingTask != null && existingTask is ExternalBuilderIncrementalTask) {
            return existingTask
        }

        return tasks.create(taskName, ExternalBuilderIncrementalTask::class.java) { t->
            t.inputDir.set(outputDir)
            t.outputDir.set(outputDir)
            t.variantName = taskVariantName
        }
    }

    private fun getExternalBuildTaskName(taskVariant: String): String {
        return "external${capitalize(taskVariant)}Build"
    }

    private fun getExternalBuildIncrementalTaskName(taskVariant: String): String {
        return "external${capitalize(taskVariant)}BuildIncremental"
    }

    private fun capitalize(str: String): String {
        return str.replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.US) else it.toString() }
    }
}
