package ru.yandex.external_builder

import com.android.builder.model.ProductFlavor
import org.apache.commons.collections4.queue.CircularFifoQueue
import org.gradle.api.DefaultTask
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.ListProperty
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.TaskAction
import org.slf4j.LoggerFactory
import java.io.BufferedReader
import java.io.File
import java.io.IOException
import java.io.InputStreamReader
import java.util.concurrent.TimeUnit

/**
 * ExternalBuilderTask performs build of Ya.Package by calling `ya package` command
 * on a given package json.
 *
 * It executes build described by [packageJson] and stores its result in [outputDir].
 * Specified [yaTool] will be used as a build tool for packages.
 */
abstract class ExternalBuilderTask : DefaultTask() {

    @get:Internal
    internal abstract val yaTool: RegularFileProperty
    @get:Internal
    internal abstract val ytCacheArgs: ListProperty<String>

    @get:Internal
    internal abstract val packageJson: RegularFileProperty
    @get:Internal
    internal abstract val externalFlavors: ListProperty<ExternalBuilderExtension.Flavor>

    @get:OutputDirectory
    abstract val outputDir: DirectoryProperty

    @get:Internal
    internal lateinit var flavors: List<ProductFlavor>

    private val logger by lazy { LoggerFactory.getLogger(name) }

    init {
        // Since our task has defined output directory, but has no input files,
        // Gradle cannot track incremental changes in inputs of this task.
        // To solve that we explicitly ask Gradle here to rerun this task each time
        // when any parent task is executed, even when all parent tasks are up to date.
        outputs.upToDateWhen { false }
    }

    @TaskAction
    fun build() {
        val startTime = System.currentTimeMillis()
        logger.info("external build started")

        try {
            doBuild()
        } catch (error: IOException) {
            throw RuntimeException("ya package run failed: ${error.message}")
        }

        val runtime = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - startTime)
        logger.info("external build finished ($runtime sec)")
    }

    @Throws(IOException::class)
    fun doBuild() {
        val yaTool = yaTool.asFile.get().canonicalPath
        val packageJson = getPackageJsonForCurrentFlavors().canonicalPath
        val output = outputDir.asFile.get()

        ensureOutputDirExists(output)
        val command = arrayListOf(yaTool, "package", packageJson, "--raw-package-path", output.name, "--build", "release") + ytCacheArgs.get()

        logger.debug("cwd: ${output.parentFile.canonicalPath}")
        logger.debug("command: [${command.joinToString(separator=", ", transform={ "\"$it\"" })}]")
        val process = ProcessBuilder(command).directory(output.parentFile).redirectErrorStream(true).start()

        // save logs in case of a possible build failure
        val savedLogs = CircularFifoQueue<String>(MAX_REMEMBERED_LOGS_LINES)
        BufferedReader(InputStreamReader(process.inputStream)).use { bufferedReader ->
            var line: String?
            while (bufferedReader.readLine().also { line = it } != null) {
                if (!logger.isInfoEnabled) {
                    savedLogs.add(line)
                } else {
                    logger.info(line)
                }
            }
        }

        val exitCode = process.waitFor()
        if (exitCode != 0) {
            if (savedLogs.isAtFullCapacity) {
                logger.warn("Logs output was truncated. Run build with --info for full output\n")
            }
            savedLogs.forEach { logger.warn(it) }
            throw RuntimeException("ya package finished with non-zero value: $exitCode")
        }
    }

    private fun ensureOutputDirExists(outputDir: File) {
        if (outputDir.exists()) {
            if (!outputDir.isDirectory) {
                throw IllegalArgumentException("output dir is not directory: ${outputDir.canonicalPath}")
            }
        } else if (!outputDir.mkdirs()) {
            throw IllegalArgumentException("cannot create output dir: ${outputDir.canonicalPath}")
        }
    }

    private fun getPackageJsonForCurrentFlavors(): File {
        var match: ExternalBuilderExtension.Flavor? = null
        for (flavor in flavors) {
            for (externalFlavor in externalFlavors.get()) {
                if (flavor.name == externalFlavor.name && flavor.dimension == externalFlavor.dimension) {
                    if (match != null) {
                        throw IllegalStateException("Conflicted flavors config was detected: " +
                                "$packageJson is conflicting with $match")
                    }
                    match = externalFlavor
                }
            }
        }

        if (match == null && externalFlavors.get().isNotEmpty()) {
            throw IllegalStateException("No matching flavor found for external plugin")
        }

        return match?.packageJson ?: packageJson.asFile.get()
    }

    companion object {
        const val MAX_REMEMBERED_LOGS_LINES = 512
    }
}
