package ru.yandex.io.sdk

import android.app.Service
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.Binder
import android.os.Build
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.system.Os
import androidx.annotation.MainThread
import androidx.annotation.WorkerThread
import com.yandex.launcher.logger.KLogger
import ru.yandex.io.common.AliceRequestCallback
import ru.yandex.alice.library.client.protos.ClientInfoProto
import ru.yandex.alice.megamind.protos.common.DeviceStateProto
import ru.yandex.alice.megamind.protos.speechkit.SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource
import ru.yandex.alice.protos.endpoint.EndpointProto
import ru.yandex.io.sdk.assets.QuasarAssetsManager
import ru.yandex.io.sdk.audio.AndroidToQuasarAudioBus
import ru.yandex.io.sdk.capability.Capability
import ru.yandex.io.sdk.jni.DeviceCryptography
import ru.yandex.io.sdk.jni.DeviceUtils
import ru.yandex.io.sdk.jni.JniAliceRequestCallback
import ru.yandex.io.sdk.jni.JniAudioClientEventObserver
import ru.yandex.io.sdk.jni.JniAuthObserver
import ru.yandex.io.sdk.jni.JniBrickStatusObserver
import ru.yandex.io.sdk.jni.JniConfigObserver
import ru.yandex.io.sdk.jni.JniDeviceGroupStateObserver
import ru.yandex.io.sdk.jni.JniDirectiveObserver
import ru.yandex.io.sdk.jni.JniGlagolDiscoveryObserver
import ru.yandex.io.sdk.jni.JniMediaObserver
import ru.yandex.io.sdk.jni.JniMusicStateObserver
import ru.yandex.io.sdk.jni.JniSDKStateObserver
import ru.yandex.io.sdk.jni.JniVinsResponsePreprocessor
import ru.yandex.io.sdk.jni.QuasarClient
import ru.yandex.io.sdk.jni.QuasarLauncher
import ru.yandex.io.sdk.jni.YandexIOSDK
import ru.yandex.io.sdk.jni.YandexIoEndpoint
import ru.yandex.io.sdk.results.CallSdkResult
import ru.yandex.io.sdk.results.RegistrationResult
import ru.yandex.io.sdk.utils.ForegroundServiceHelper
import ru.yandex.io.sdk.utils.ProcessUtils
import ru.yandex.quasar.protobuf.DeviceSdkProto
import ru.yandex.quasar.protobuf.ModelObjects
import ru.yandex.quasar.protobuf.YandexIO.DirectiveEntity
import ru.yandex.quasar.protobuf.YandexIO.TvPolicyInfo
import ru.yandex.quasar.scaffolding.proto.Config
import java.util.concurrent.Callable
import java.util.concurrent.CountDownLatch
import java.util.concurrent.Executors
import java.util.concurrent.ForkJoinPool
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit
import kotlin.system.exitProcess

/**
 * Base IoService.
 */
//  Keep-alive logic should be kept consistent with Alice KeepAliceService
//  https://a.yandex-team.ru/arc_vcs/smart_devices/android/tv/alice-app/src/main/java/com/yandex/tv/alice/app/KeepAliveService.java
abstract class YandexIoService : Service() {
    private val foregroundServiceHelper = ForegroundServiceHelper(NOTIFICATION_CHANNEL_ID, NOTIFICATION_CHANNEL_NAME)

    private var ioSDK: YandexIOSDK? = null

    private val quasarLauncher = QuasarLauncher()

    private var isQuasarLaunched = false  // synchronized on this
    private val quasarLaunchLatch = CountDownLatch(1)

    private var audioBus: AndroidToQuasarAudioBus? = null
    private var jniDirectiveObserver: JniDirectiveObserver? = null
    private var jniMediaObserver: JniMediaObserver? = null
    private var jniMusicStateObserver: JniMusicStateObserver? = null
    private var jniAudioClientEventObserver: JniAudioClientEventObserver? = null
    private var jniSDKStateObserver: JniSDKStateObserver? = null
    private var jniAuthObserver: JniAuthObserver? = null
    private var jniConfigObserver: JniConfigObserver? = null
    private var jniVinsResponsePreprocessorHandler: JniVinsResponsePreprocessor? = null
    private var jniBrickStatusObserver: JniBrickStatusObserver? = null
    private var jniDeviceGroupStateObserver: JniDeviceGroupStateObserver? = null
    private var jniGlagolDiscoveryObserver: JniGlagolDiscoveryObserver? = null
    private var deviceCryptography: DeviceCryptography? = null
    private var deviceUtils: DeviceUtils? = null
    private var capabilitiesEndpoint: Lazy<YandexIoEndpoint>? = null

    // Binder given to clients
    private val binder = LocalBinder()

    // Older apis, since Executors.newWorkStealingPool is not available
    protected val executor = ForkJoinPool(
        Runtime.getRuntime().availableProcessors(),
        ForkJoinPool.defaultForkJoinWorkerThreadFactory,
        null, true
    )

    private val quasarLauncherExecutor = Executors.newSingleThreadExecutor()
    private val mainHandler = Handler(Looper.getMainLooper())

    abstract fun createDeviceInfo(): DeviceInfo

    abstract fun createIoSdkSettings(): IoSdkSettings

    override fun onCreate() {
        // start as "foreground" service asap
        doStartForeground()
        KLogger.d(TAG) { "onCreate" }
        loadQuasarNativeLibs()
        initJni()
    }

    // used to start iosdk without binders and services
    protected fun startQuasarInternal() {
        KLogger.d(TAG) { "startQuasarInternal" }
        loadQuasarNativeLibs()
        initJni()
        launchQuasarDaemons(null)
    }

    private fun initJni() {
        KLogger.d(TAG) { "initJni" }
        jniDirectiveObserver = JniDirectiveObserver()
        jniMediaObserver = JniMediaObserver()
        jniMusicStateObserver = JniMusicStateObserver()
        jniAudioClientEventObserver = JniAudioClientEventObserver()
        jniSDKStateObserver = JniSDKStateObserver()
        jniAuthObserver = JniAuthObserver()
        jniConfigObserver = JniConfigObserver()
        jniVinsResponsePreprocessorHandler = JniVinsResponsePreprocessor()
        jniBrickStatusObserver = JniBrickStatusObserver()
        jniDeviceGroupStateObserver = JniDeviceGroupStateObserver()
        jniGlagolDiscoveryObserver = JniGlagolDiscoveryObserver()
        deviceCryptography = DeviceCryptography()
        deviceUtils = DeviceUtils()
        capabilitiesEndpoint = lazy {
            YandexIoEndpoint(
                createDeviceInfo().quasarDeviceId,
                EndpointProto.TEndpoint.EEndpointType.SpeakerEndpointType
            )
        }
        KLogger.d(TAG) { "initJni done" }
    }

    private fun resetJni() {
        KLogger.d(TAG) { "resetJni" }
        jniDirectiveObserver?.reset()
        jniMediaObserver?.reset()
        jniMusicStateObserver?.reset()
        jniAudioClientEventObserver?.reset()
        jniSDKStateObserver?.reset()
        jniAuthObserver?.reset()
        jniConfigObserver?.reset()
        jniBrickStatusObserver?.reset()
        jniDeviceGroupStateObserver?.reset()
        jniGlagolDiscoveryObserver?.reset()

        jniVinsResponsePreprocessorHandler = null
        deviceCryptography = null
        deviceUtils = null
        capabilitiesEndpoint = null
        KLogger.d(TAG) { "resetJni done" }
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        KLogger.d(TAG) { "onStartCommand($intent, $flags, $startId)" }
        val res = super.onStartCommand(intent, flags, startId)

        when (intent?.action) {
            ACTION_RUN_QUASAR_CLIENT -> runQuasarClient(intent)
            else -> launchQuasarDaemons(intent)
        }
        return res
    }

    private fun doStartForeground() {
        foregroundServiceHelper.startForeground(this)
    }

    private fun runQuasarClient(intent: Intent) {
        val service = intent.getStringExtra("service")
        val message = intent.getStringExtra("message")
        quasarLauncherExecutor.execute {
            KLogger.i(TAG) { "Send message to `$service`, message=$message" }
            QuasarClient().sendQuasarMessage(service, message)
        }
    }

    @MainThread
    private fun launchQuasarDaemons(intent: Intent?) {
        KLogger.d(TAG) { "launchQuasarDaemons on ${Thread.currentThread().name}" }
        quasarLauncherExecutor.execute {
            KLogger.d(TAG) { "launchQuasarDaemons on ${Thread.currentThread().name}" }
            synchronized(this) {
                if (isQuasarLaunched) {
                    KLogger.w(TAG) { "Quasar is already started" }
                    return@execute
                }

                val sdkSettings = createIoSdkSettings()
                startQuasar(intent?.getStringExtra("authCode"), sdkSettings)
                ioSDK = YandexIOSDK()
                jniDirectiveObserver?.register()
                jniMediaObserver?.register()
                jniMusicStateObserver?.register()
                jniAudioClientEventObserver?.register()
                jniSDKStateObserver?.register()
                jniAuthObserver?.register()
                jniConfigObserver?.register()
                jniVinsResponsePreprocessorHandler?.register()
                jniBrickStatusObserver?.register()
                jniDeviceGroupStateObserver?.register()
                jniGlagolDiscoveryObserver?.register()

                audioBus = AndroidToQuasarAudioBus()
                isQuasarLaunched = true
                onQuasarStarted()
            }
            quasarLaunchLatch.countDown()
        }
    }

    override fun onDestroy() {
        KLogger.d(TAG) { "onDestroy" }
        synchronized(this) {
            if (isQuasarLaunched) {
                isQuasarLaunched = false
                executor.submit {
                    stopQuasar()
                    resetJni()
                    executor.shutdownNow()
                }
                quasarLauncherExecutor.shutdownNow()
                ioSDK = null
                audioBus = null
            }
        }
        super.onDestroy()
    }

    /* Call jni method to toggle conversation with Alice */

    fun toggleConversation() {
        callSdkSafely("toggleConversation") { toggleConversation() }
    }

    fun toggleConversation(eventSource: TEventSource) {
        callSdkSafely("toggleConversation") { toggleConversation(eventSource.toByteArray()) }
    }

    fun startConversation() {
        callSdkSafely("startConversation") { startConversation() }
    }

    fun startConversation(eventSource: TEventSource) {
        callSdkSafely("startConversation") { startConversation(eventSource.toByteArray()) }
    }

    fun stopConversation() {
        callSdkSafely("stopConversation") { stopConversation() }
    }

    fun stopAlarm() {
        callSdkSafely("stopAlarm") { stopAlarm() }
    }

    fun finishConversationVoiceInput() {
        callSdkSafely("finishConversationVoiceInput") { finishConversationVoiceInput() }
    }

    fun registerDeviceInBackend(oauthToken: String, uid: String) {
        callSdkSafely("registerDeviceInBackend") { registerDeviceInBackend(oauthToken, uid) }
    }

    fun registerDeviceInBackendSync(
        oauthToken: String,
        uid: String
    ): Future<CallSdkResult<RegistrationResult?>> {
        return callSdkSafelyWithResult("registerDeviceInBackendSync") {
            registerDeviceInBackendSync(oauthToken, uid)
        }
    }

    fun provideUserAccountInfo(oauthToken: String, uid: String) {
        callSdkSafely("provideUserAccountInfo") { provideUserAccountInfo(oauthToken, uid) }
    }

    fun revokeUserAccountInfo() {
        callSdkSafely("revokeUserAccountInfo") { revokeUserAccountInfo() }
    }

    /* Call jni method to send text command to Alice */
    fun sendTextQuery(query: String, callback: AliceRequestCallback? = null, eventSource: TEventSource? = null) {
        val jniCallback = callback?.let { JniAliceRequestCallback(mainHandler, callback) }
        if (eventSource == null) {
            callSdkSafely("sendTextQuery") { sendTextQuery(query, jniCallback) }
        } else {
            callSdkSafely("sendTextQuery") { sendTextQuery(query, jniCallback, eventSource.toByteArray()) }
        }
    }

    fun sendUnstructuredServerRequest(
        request: String,
        enqueued: Boolean = false,
        parallel: Boolean = false,
        callback: AliceRequestCallback? = null,
        eventSource: TEventSource? = null
    ) {
        val jniCallback = callback?.let { JniAliceRequestCallback(mainHandler, callback) }
        if (eventSource == null) {
            callSdkSafely("sendUnstructuredServerRequest") {
                sendUnstructuredServerRequest(request, enqueued, parallel, jniCallback)
            }
        } else {
            callSdkSafely("sendUnstructuredServerRequest") {
                sendUnstructuredServerRequest(request, enqueued, parallel, jniCallback, eventSource.toByteArray())
            }
        }
    }

    fun pause() {
        callSdkSafely("pause") { pause() }
    }

    fun togglePlayPause() {
        callSdkSafely("togglePlayPause") { togglePlayPause() }
    }

    fun nextTrack() {
        callSdkSafely("nextTrack") { nextTrack() }
    }

    fun prevTrack() {
        callSdkSafely("prevTrack") { prevTrack() }
    }

    fun like() {
        callSdkSafely("like") { like() }
    }

    fun dislike() {
        callSdkSafely("dislike") { dislike() }
    }

    fun authenticate(xcode: String) {
        callSdkSafely("authenticate") { authenticate(xcode) }
    }

    @JvmOverloads
    fun playSoundFile(fileName: String, channel: ModelObjects.AudioChannel?, playLooped: Boolean = false) {
        callSdkSafely("playSoundFile") { playSoundFile(fileName, channel, playLooped) }
    }

    fun stopSoundFile(fileName: String) {
        callSdkSafely("stopSoundFile") { stopSoundFile(fileName) }
    }

    fun sendAudioRewind(amountMs: Double, rewindType: String) {
        callSdkSafely("sendAudioRewind") { sendAudioRewind(amountMs, rewindType) }
    }

    fun sendActiveActions(activeActions: DeviceStateProto.TDeviceState.TActiveActions) {
        callSdkSafely("sendActiveActions") { sendActiveActions(activeActions) }
    }

    fun sendActiveActionSemanticFrame(activeAction: String?) {
        callSdkSafely("sendActiveActionSemanticFrame") { sendActiveActionSemanticFrame(activeAction) }
    }

    fun sendTvPolicyInfo(tvPolicyInfo: TvPolicyInfo) {
        callSdkSafely("sendTvPolicyInfo") { sendTvPolicyInfo(tvPolicyInfo) }
    }

    fun provideDeviceState(statePart: DeviceSdkProto.TDeviceStatePart) {
        callSdkSafely("provideDeviceState") { provideState(statePart) }
    }

    fun provideMediaDeviceIdentifier(identifier:  ClientInfoProto.TClientInfoProto.TMediaDeviceIdentifier) {
        callSdkSafely("provideMediaDeviceIdentifier") { provideMediaDeviceIdentifier(identifier) }
    }

    fun sendWifiList(wifiList: List<ModelObjects.WifiInfo>) {
        callSdkSafely("sendWifiList") { sendWifiList(wifiList) }
    }

    fun sendTimezone(timezone: String, offsetSec: Int){
        callSdkSafely("sendTimezone") { sendTimezone(timezone, offsetSec) }
    }

    fun sendLocation(latitude: Double, longitude: Double, accuracy: Double?) {
        callSdkSafely("sendLocation") { sendLocation(latitude, longitude, accuracy) }
    }

    fun sendIsMicrophoneMuted(muted: Boolean) {
        callSdkSafely("sendIsMicrophoneMuted") { sendIsMicrophoneMuted(muted) }
    }

    fun decryptBlePlainTextCredentials(plainText: String): ModelObjects.BleDecryptCredentialsStatus? =
        deviceCryptography?.decryptBlePlainTextCredentials(plainText)

    fun decryptBleCredentials(
        credentials: ModelObjects.EncryptedSetupCredentialsMessage
    ): ModelObjects.BleDecryptCredentialsStatus? = deviceCryptography?.decryptBleCredentials(credentials)

    fun prepareSignedHeaders(plainText: String) = deviceCryptography?.prepareSignedHeaders(plainText) ?: emptyMap()

    fun calcSsidHashcode(ssid: String) = deviceUtils?.calcSsidHashcode(ssid)

    private abstract class IoSdkAction(private val actionName: String) : Runnable {
        override fun toString(): String {
            return actionName
        }
    }

    private abstract class IoSdkActionCallable<T>(private val actionName: String) : Callable<T> {
        override fun toString(): String {
            return actionName
        }
    }

    fun callSdkSafely(actionName: String, sdkAction: YandexIOSDK.() -> Unit) {
        val action = object : IoSdkAction(actionName) {
            override fun run() {
                KLogger.d(TAG) { "execute $actionName on ${Thread.currentThread().name}" }
                try {
                    if (quasarLaunchLatch.await(QUASAR_LAUNCH_TIMEOUT_SEC, TimeUnit.SECONDS)) {
                        ioSDK?.sdkAction()
                        KLogger.d(TAG) { "complete $actionName" }
                    } else {
                        KLogger.e(TAG) { "$actionName failed: iosdk have not launched in $QUASAR_LAUNCH_TIMEOUT_SEC sec" }
                    }
                } catch (e: Exception) {
                    KLogger.e(TAG, e) { "$actionName failed" }
                }
            }
        }
        executor.execute(action)
    }

    @Suppress("SameParameterValue")
    private fun <T> callSdkSafelyWithResult(actionName: String, sdkAction: YandexIOSDK.() -> CallSdkResult<T?>): Future<CallSdkResult<T?>> {
        val action = object : IoSdkActionCallable<CallSdkResult<T?>>(actionName) {
            override fun call(): CallSdkResult<T?> {
                KLogger.d(TAG) { "execute $actionName on ${Thread.currentThread().name}" }
                return try {
                    if (quasarLaunchLatch.await(QUASAR_LAUNCH_TIMEOUT_SEC, TimeUnit.SECONDS)) {
                        val result = ioSDK!!.sdkAction()
                        KLogger.d(TAG) { "complete $actionName" }
                        result
                    } else {
                        KLogger.e(TAG) { "$actionName failed: iosdk have not launched after $QUASAR_LAUNCH_TIMEOUT_SEC sec" }
                        CallSdkResult(false, null)
                    }
                } catch (e: Exception) {
                    KLogger.e(TAG, e) { "$actionName failed" }
                    CallSdkResult(false, null)
                }
            }
        }
        return executor.submit(action)
    }

    open fun startAudioRecording() {
        audioBus?.start()
    }

    fun stopAudioRecording() {
        audioBus?.stop()
    }

    fun sendIsScreenActive(isScreenActive: Boolean) {
        callSdkSafely("sendIsScreenActive") { sendIsScreenActive(isScreenActive) }
    }

    fun executeDirectiveSequence(directiveEntityList: List<DirectiveEntity>) {
        callSdkSafely("sendDirectivesResponse") {
            executeDirectiveSequence(directiveEntityList)
        }
    }

    fun addCapability(capability: Capability) {
        capabilitiesEndpoint?.value?.addCapability(capability)
    }

    fun removeCapability(capability: Capability) {
        capabilitiesEndpoint?.value?.removeCapability(capability)
    }

    @MainThread
    fun addDirectiveObserver(directiveObserver: DirectiveObserver) {
        jniDirectiveObserver?.addObserver(directiveObserver)
    }

    @MainThread
    fun removeDirectiveObserver(directiveObserver: DirectiveObserver) {
        jniDirectiveObserver?.removeObserver(directiveObserver)
    }

    @MainThread
    fun addMediaObserver(mediaObserver: MediaObserver) {
        jniMediaObserver?.addObserver(mediaObserver)
    }

    @MainThread
    fun removeMediaObserver(mediaObserver: MediaObserver) {
        jniMediaObserver?.removeObserver(mediaObserver)
    }

    @MainThread
    fun addMusicStateObserver(musicStateObserver: MusicStateObserver) {
        jniMusicStateObserver?.addObserver(musicStateObserver)
    }

    @MainThread
    fun removeMusicStateObserver(musicStateObserver: MusicStateObserver) {
        jniMusicStateObserver?.removeObserver(musicStateObserver)
    }

    @MainThread
    fun addAudioClientEventObserver(audioClientEventObserver: AudioClientEventObserver) {
        jniAudioClientEventObserver?.addObserver(audioClientEventObserver)
    }

    @MainThread
    fun removeAudioClientEventObserver(audioClientEventObserver: AudioClientEventObserver) {
        jniAudioClientEventObserver?.removeObserver(audioClientEventObserver)
    }

    @MainThread
    fun addDeviceGroupStateObserver(deviceGroupStateObserver: DeviceGroupStateObserver) {
        jniDeviceGroupStateObserver?.addObserver(deviceGroupStateObserver)
    }

    @MainThread
    fun removeDeviceGroupStateObserver(deviceGroupStateObserver: DeviceGroupStateObserver) {
        jniDeviceGroupStateObserver?.removeObserver(deviceGroupStateObserver)
    }

    @MainThread
    fun addGlagolDiscoveryObserver(glagolDiscoveryObserver: GlagolDiscoveryObserver) {
        jniGlagolDiscoveryObserver?.addObserver(glagolDiscoveryObserver)
    }

    @MainThread
    fun removeGlagolDiscoveryObserver(glagolDiscoveryObserver: GlagolDiscoveryObserver) {
        jniGlagolDiscoveryObserver?.removeObserver(glagolDiscoveryObserver)
    }

    @MainThread
    fun addSDKStateObserver(sdkStateObserver: SDKStateObserver) {
        jniSDKStateObserver?.addObserver(sdkStateObserver)
    }

    @MainThread
    fun removeSDKStateObserver(sdkStateObserver: SDKStateObserver) {
        jniSDKStateObserver?.removeObserver(sdkStateObserver)
    }

    @MainThread
    fun addAuthObserver(authObserver: AuthObserver) {
        jniAuthObserver?.addObserver(authObserver)
    }

    @MainThread
    fun removeAuthObserver(authObserver: AuthObserver) {
        jniAuthObserver?.removeObserver(authObserver)
    }

    @MainThread
    fun addBrickStatusObserver(brickStatusObserver: BrickStatusObserver) {
        jniBrickStatusObserver?.addObserver(brickStatusObserver)
    }

    @MainThread
    fun removeBrickStatusObserver(brickStatusObserver: BrickStatusObserver) {
        jniBrickStatusObserver?.removeObserver(brickStatusObserver)
    }

    fun setVinsResponsePreprocessorHandler(
        handler: JniVinsResponsePreprocessor.Handler
    ) {
        jniVinsResponsePreprocessorHandler?.setHandler(handler)
    }

    @MainThread
    fun subscribeToConfig(configObserver: ConfigObserver, configNames: Array<String>) {
        callSdkSafely("subscribeToSystemConfig") { jniConfigObserver?.subscribeToConfig(configObserver, configNames) }
    }

    @MainThread
    fun unsubscribeFromConfig(configObserver: ConfigObserver) {
        callSdkSafely("unsubscribeFromSystemConfig") { jniConfigObserver?.unsubscribeFromConfig(configObserver) }
    }

    override fun onBind(intent: Intent?): IBinder {
        return binder
    }

    @WorkerThread
    private fun startQuasar(authCode: String?, sdkSettings: IoSdkSettings) {
        KLogger.i(TAG) { "Start Quasar service at $filesDir" }

        val dirs = QuasarDirs(this)
        val qam = QuasarAssetsManager(this)
        qam.migrateAssetsOnAppVersionUpgrade()

        val deviceInfo = createDeviceInfo()

        // Prepare config for start,
        // for placeholders refer to patterns in https://a.yandex-team.ru/arc_vcs/yandex_io/android_sdk/config/android.cfg#L2
        val configBuilder = Config.LauncherConfig.newBuilder()
        configBuilder.pathsBuilder
            .setWorkdirPath(dirs.workDir.toString())
            .setConfigPath(qam.getQuasarConfig().path)
            .putConfigPlaceholders("QUASAR", dirs.systemDir.toString())
            .putConfigPlaceholders("FILES", this.filesDir.toString())
            .putConfigPlaceholders("DATA", dirs.dataDir.toString())
            .putConfigPlaceholders("APP_ID", deviceInfo.appId)
            .putConfigPlaceholders("APP_VERSION", deviceInfo.appVersion)
            .putConfigPlaceholders("SOFTWARE_VERSION", deviceInfo.appVersion)
            .putConfigPlaceholders("OS", deviceInfo.os)
            .putConfigPlaceholders("OS_VERSION", deviceInfo.osVersion)
            .putConfigPlaceholders("DEVICE_TYPE", deviceInfo.deviceType)
            .putConfigPlaceholders("DEVICE_MANUFACTURER", deviceInfo.deviceManufacturer)
            .putConfigPlaceholders("USER_AGENT", sdkSettings.getUserAgent(deviceInfo))
            .putConfigPlaceholders("CRYPTOGRAPHY_TYPE", sdkSettings.cryptographyType.sdkName)
            .putConfigPlaceholders("LOG_LEVEL", sdkSettings.logLevel.sdkName)
            .putConfigPlaceholders("PLAIN_FILE_RSA_PADDING", sdkSettings.plainFileRsaPadding.value.toString())

        getQuasarBackend()?.let {
            configBuilder.pathsBuilder.putConfigPlaceholders("QUASAR_BACKEND", it)
        }

        authCode?.let {
            configBuilder.authCode = it
        }

        configBuilder.deviceId = deviceInfo.quasarDeviceId
        configBuilder.limitsBuilder.filenoLimit = 4096
        configBuilder.disableBackgroundServices = sdkSettings.disableBackgroundServices
        configBuilder.useNetworkIpc = sdkSettings.useNetworkIpc

        val config = configBuilder.build()

        // Prepare environment
        Os.setenv("CA_CERTIFICATES", qam.getCaCertificates().path, true)

        KLogger.d(TAG) { "Quasar launch with $config" }
        quasarLauncher.start(config)
        onQuasarStart()
    }

    open fun getQuasarBackend(): String? = null

    /**
     * @note run when quasar is about to start
     */
    @WorkerThread
    protected open fun onQuasarStart() {
        // to be overridden if needed
    }

    /**
     * @note run when quasar is finished startup procedures
     */
    @WorkerThread
    protected open fun onQuasarStarted() {
        // to be overridden if needed
    }

    @WorkerThread
    private fun stopQuasar() {
        KLogger.i(TAG) { "Stop Quasar service" }
        onQuasarStop()
        quasarLauncher.stop()
    }

    @WorkerThread
    protected open fun onQuasarStop() {
        // to be overridden if needed
    }

    private fun loadQuasarNativeLibs() {
        try {
            Runtime.getRuntime().loadLibrary("quasar_daemons")
        } catch (e: UnsatisfiedLinkError) {
            System.err.println("Native code library failed to load.\n$e")
            exitProcess(1)
        }

        KLogger.i(TAG) { "Quasar libs are loaded!" }
    }

    /**
     * Class used for the client Binder.  Because we know this service always
     * runs in the same process as its clients, we don't need to deal with IPC.
     */
    inner class LocalBinder : Binder() {
        // Return this instance of QuasarService so clients can call public methods
        fun getService(): YandexIoService = this@YandexIoService
    }

    companion object {
        private const val TAG = "YandexIoService"
        private const val ACTION_RUN_QUASAR_CLIENT = "ru.yandex.quasar.io.sdk.RUN_QUASAR_CLIENT"

        private const val NOTIFICATION_CHANNEL_ID = "YandexIoService"
        private const val NOTIFICATION_CHANNEL_NAME = "Infrastructure"

        private const val QUASAR_LAUNCH_TIMEOUT_SEC = 120L

        fun <T : YandexIoService> wakeUp(context: Context, serviceClass: Class<T>) {
            // avoiding ANR
            // * E/ActivityManager: ANR in com.yandex.tv.alice
            // * Reason: Context.startForegroundService() did not then call Service.startForeground
            // *   (): ServiceRecord{8014e56 u0 com.yandex.tv.alice/.app.KeepAliveService}
            //
            // with more reliable but hacky solution:
            // based on SO answer: https://stackoverflow.com/a/57521350
            if (ProcessUtils.isRunningInForeground(context, serviceClass) == true) {
                KLogger.d(TAG) { "wakeUp call is skipped since service already in foreground" }
                return
            }
            val intent = Intent(context, serviceClass)
            val startServiceFromContext = Runnable {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    context.startForegroundService(intent)
                } else {
                    context.startService(intent)
                }
            }

            // Create the service connection.
            val connection: ServiceConnection = object : ServiceConnection {
                override fun onServiceConnected(name: ComponentName, service: IBinder) {
                    KLogger.w(TAG) { "wakeUp onServiceConnected" }
                    val yandexIoService =
                        serviceClass.cast((service as LocalBinder).getService())

                    // note: we should NOT skip this block of code via !isRunningInForeground(context) condition
                    // it will cause service to stop in some cases
                    run {
                        startServiceFromContext.run()
                        // This is the key: Without waiting Android Framework to call this method
                        // inside Service.onCreate() .onStartCommand(), immediately call here to post the notification.
                        yandexIoService?.doStartForeground()
                    }
                    context.unbindService(this)
                    KLogger.w(TAG) { "wakeUp onServiceConnected finished" }
                }

                override fun onBindingDied(name: ComponentName) {
                    KLogger.w(TAG) { "wakeUp Binding has dead" }
                }

                override fun onNullBinding(name: ComponentName) {
                    KLogger.w(TAG) { "wakeUp Bind was null" }
                }

                override fun onServiceDisconnected(name: ComponentName) {
                    KLogger.w(TAG) { "wakeUp Service is disconnected" }
                }
            }

            // Try to bind the service
            try {
                context.bindService(intent, connection, BIND_AUTO_CREATE)
            } catch (e: Throwable) {
                KLogger.e(TAG, e) { "wakeUp failed to bind to service" }
                // fallback to plain start from context
                startServiceFromContext.run()
            }
        }
    }
}
