package ru.yandex.io.sdk.assets

import android.content.Context
import android.content.pm.PackageManager
import androidx.annotation.WorkerThread
import androidx.core.content.edit
import com.yandex.launcher.logger.KLogger

/**
 * This class is used to check whether update of assets is required or not.
 * It triggers migration if the application's version code was changed or
 * the application is installed for the first time.
 */
class AssetsVersionHelper(private val context: Context) {
    private val prefs by lazy { context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE) }

    @WorkerThread
    fun isAssetsMigrationRequired(): Boolean {
        val packageVersion = getPackageVersion()
        val assetsVersion = getAssetsVersion()
        KLogger.d(TAG) { "Package version: $packageVersion, assets version: $assetsVersion" }
        return packageVersion != assetsVersion
    }

    @WorkerThread
    fun commitAssetsMigration() = setAssetsVersion(getPackageVersion())

    private fun setAssetsVersion(version: Long) {
        prefs.edit(commit = true) { putLong(KEY_LAST_UNPACKED_VERSION, version) }
    }

    private fun getAssetsVersion(): Long {
        return prefs.getLong(KEY_LAST_UNPACKED_VERSION, -1)
    }

    private fun getPackageVersion(): Long {
        val packageInfo = try {
            context.packageManager.getPackageInfo(context.packageName, 0)
        } catch (ex: PackageManager.NameNotFoundException) {
            KLogger.e(TAG, ex) { "Error getting package info" }
            return 0
        }
        return if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.P) {
            packageInfo.longVersionCode
        } else {
            @Suppress("DEPRECATION")
            packageInfo.versionCode.toLong()
        }
    }

    private companion object {
        const val TAG = "AssetsVersionHelper"
        const val PREFS_NAME = "assets_version"
        const val KEY_LAST_UNPACKED_VERSION = "last_unpacked_version"
    }
}
