package ru.yandex.io.sdk.assets

import android.content.Context
import androidx.annotation.AnyThread
import androidx.annotation.WorkerThread
import com.yandex.launcher.logger.KLogger
import ru.yandex.io.sdk.QuasarDirs
import java.io.BufferedOutputStream
import java.io.File
import java.io.FileOutputStream
import java.lang.IllegalStateException

/**
 * This class performs unpacking of assets from .apk file to working directory of Quasar Daemons.
 *
 * Unpacking happens when application is installed for the first time or the application's
 * version code was changed since last unpacking.
 *
 * If application's version code is changed, old assets will be deleted before unpacking the new ones.
 */
class QuasarAssetsManager(private val context: Context,
                          private val assetsLocationOnDisk: File = QuasarDirs(context).systemDir) {

    private val versionHelper = AssetsVersionHelper(context)

    @AnyThread
    fun getQuasarConfig() = File(assetsLocationOnDisk, "quasar.cfg")

    @AnyThread
    fun getCaCertificates() = File(assetsLocationOnDisk, "ca-certificates.crt")

    @WorkerThread
    fun migrateAssetsOnAppVersionUpgrade() {
        if (versionHelper.isAssetsMigrationRequired() || !getQuasarConfig().exists()) {
            KLogger.d(TAG) { "Running assets migration" }
            deleteExistingAssets(assetsLocationOnDisk)
            unpackAsset(QUASAR_ASSETS_DIR_IN_APK, assetsLocationOnDisk)
            versionHelper.commitAssetsMigration()
        }
    }

    private fun unpackAsset(src: String, dest: File) {
        // Checking whether src is directory or file
        val directoryContent = context.assets.list(src) ?: emptyArray()
        if (directoryContent.isEmpty()) {
            // Assuming that src is a file
            KLogger.d(TAG) { "Unpacking file $src to $dest" }
            unpackAssetFile(src, dest)
        } else {
            // Assuming that src is a directory
            KLogger.d(TAG) { "Unpacking directory $src to $dest" }
            unpackAssetDir(src, dest, directoryContent)
        }
    }

    private fun unpackAssetFile(src: String, dest: File) {
        try {
            context.assets.open(src).use { inputStream ->
                BufferedOutputStream(FileOutputStream(dest)).use { outputStream ->
                    inputStream.copyTo(outputStream)
                }
            }
        } catch (e: Exception) {
            KLogger.w(TAG, e) { "Failed to unpack file '$src' to '$dest'" }
        }
    }

    private fun unpackAssetDir(src: String, dest: File, directoryContent: Array<String>) {
        if (!dest.isDirectory) {
            KLogger.d(TAG) { "Directory $dest does not exist: creating" }
            if (!dest.mkdirs()) {
                KLogger.e(TAG) { "Failed to create directory at '$dest'" }
                return
            }
        }
        for (assetName in directoryContent) {
            unpackAsset(File(src, assetName).path, File(dest, assetName))
        }
    }

    private fun deleteExistingAssets(unpackedAssetsDirectory: File) {
        if (unpackedAssetsDirectory.exists()) {
            KLogger.d(TAG) { "Deleting existing assets" }
            if (!unpackedAssetsDirectory.deleteRecursively()) {
                throw IllegalStateException("Cannot delete existing assets $unpackedAssetsDirectory")
            }
        }
    }

    companion object {
        const val TAG = "AssetsManager"
        const val QUASAR_ASSETS_DIR_IN_APK = "quasar"
    }
}
