package ru.yandex.io.sdk.audio

import android.annotation.SuppressLint
import android.media.AudioFormat
import android.media.AudioRecord
import android.media.MediaRecorder
import android.os.Process
import com.yandex.launcher.logger.KLogger
import ru.yandex.io.sdk.jni.AudioSource
import ru.yandex.io.sdk.utils.ProcessUtils
import java.nio.ByteBuffer
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import kotlin.math.max

/**
 * Based on https://a.yandex-team.ru/arc/trunk/arcadia/speechkit/android/libspeechkit/src/main/java/ru/yandex/speechkit/internal/BaseAudioSource.java
 * May be replaced in further work
 */
@SuppressLint("MissingPermission")
internal class AndroidAudioSource(private val listener: Listener) {
    private var recordLoop: RecordLoop? = null

    @Synchronized
    fun start() {
        KLogger.d(TAG) { "start" }
        if (recordLoop != null) {
            stop()
        }
        recordLoop = RecordLoop().apply { start() }
    }

    @Synchronized
    fun stop() {
        KLogger.d(TAG) { "stop" }
        recordLoop?.apply {
            stopRecoding()
            join()
            KLogger.d(TAG) { "RecordLoop was joined successfully" }
        }
        recordLoop = null
    }

    inner class RecordLoop : Thread("AndroidAudioSource.RecordLoop-${RECORD_LOOP_ID.incrementAndGet()}") {
        private lateinit var audioRecord: AudioRecord
        private lateinit var readBuffer: ByteBuffer

        private val isRecording = AtomicBoolean(true)

        override fun run() {
            KLogger.d(TAG) { "RecordLoop is running" }
            try {
                ProcessUtils.tryRaiseThreadPriority(Process.THREAD_PRIORITY_AUDIO)
                init()
                record()
            } catch (t: Throwable) {
                listener.onError(t)
            }
        }

        fun stopRecoding() {
            KLogger.d(TAG) { "stopRecording" }
            isRecording.set(false)
        }

        private fun init() {
            audioRecord = AudioRecord.Builder()
                .setAudioSource(AUDIO_SOURCE)
                .setAudioFormat(
                    AudioFormat.Builder()
                        .setEncoding(AUDIO_FORMAT)
                        .setSampleRate(SAMPLE_RATE_HZ)
                        .setChannelMask(CHANNEL_POSITION_MASK)
                        .build()
                ).build()


            val minMonoBufferSize = AudioRecord.getMinBufferSize(audioRecord.sampleRate, AudioFormat.CHANNEL_IN_MONO, audioRecord.audioFormat)
            if (minMonoBufferSize < 0) {
                throw Exception("getMinBufferSize error: ${getErrorText(minMonoBufferSize)} $minMonoBufferSize")
            }
            val minBufferSize = minMonoBufferSize * audioRecord.channelCount

            readBuffer = ByteBuffer.allocateDirect(minBufferSize)
        }

        private fun record() {
            startRecording()
            while (isRecording.get()) {
                val readLength = audioRecord.read(readBuffer, readBuffer.capacity())
                if (readLength < 0) {
                    throw Exception("read error: ${getErrorText(readLength)} ($readLength)")
                }
                listener.onNext(readBuffer, audioRecord.format)
                readBuffer.clear()
            }
            audioRecord.stop()
            audioRecord.release()
        }

        private fun startRecording() {
            KLogger.d(TAG) { "startRecording" }
            if (audioRecord.recordingState == AudioRecord.RECORDSTATE_RECORDING) {
                return
            }

            var durationMs = 0L
            var recordingState = AudioRecord.RECORDSTATE_STOPPED
            while (durationMs <= MICROPHONE_AVAILABILITY_CHECK_TIMEOUT_MS) {
                audioRecord.startRecording()
                recordingState = audioRecord.recordingState
                if (recordingState == AudioRecord.RECORDSTATE_RECORDING) {
                    return
                }
                durationMs += MICROPHONE_AVAILABILITY_CHECK_INTERVAL
                if (durationMs <= MICROPHONE_AVAILABILITY_CHECK_TIMEOUT_MS) {
                    KLogger.d(TAG) { "Microphone is not available. Will retry in $MICROPHONE_AVAILABILITY_CHECK_INTERVAL ms" }
                    sleep(MICROPHONE_AVAILABILITY_CHECK_INTERVAL)
                }
            }
            throw Exception("audioRecord.startRecording(), recordingState=$recordingState, durationMs=$durationMs}")
        }

        private fun getErrorText(audioRecordError: Int) : String {
            return when (audioRecordError) {
                AudioRecord.ERROR -> "ERROR"
                AudioRecord.ERROR_BAD_VALUE -> "ERROR_BAD_VALUE"
                AudioRecord.ERROR_DEAD_OBJECT -> "ERROR_DEAD_OBJECT"
                AudioRecord.ERROR_INVALID_OPERATION -> "ERROR_INVALID_OPERATION"
                else -> "Unknown error"
            }
        }
    }

    interface Listener {
        fun onNext(audioBuffer: ByteBuffer, audioFormat: AudioFormat)
        fun onError(throwable: Throwable)
    }

    private companion object {
        const val TAG = "AndroidAudioSource"

        const val SAMPLE_RATE_HZ = AudioSource.REQUIRED_SAMPLE_RATE
        const val CHANNEL_INDEX_MASK = 0xff
        const val CHANNEL_POSITION_MASK = AudioFormat.CHANNEL_IN_MONO
        const val AUDIO_FORMAT = AudioFormat.ENCODING_PCM_16BIT
        const val AUDIO_SOURCE = MediaRecorder.AudioSource.MIC

        const val MICROPHONE_AVAILABILITY_CHECK_TIMEOUT_MS = 100L
        const val MICROPHONE_AVAILABILITY_CHECK_INTERVAL = 100L

        val RECORD_LOOP_ID = AtomicInteger(0)
    }
}
