package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import androidx.annotation.MainThread;

import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.launcher.logger.Logger;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import ru.yandex.io.sdk.BrickStatusObserver;

import ru.yandex.quasar.protobuf.ModelObjects;

public class JniBrickStatusObserver {
    private static final String TAG = "JniBrickStatusObserver";
    private final Handler mainHandler = new Handler(Looper.getMainLooper());
    private final List<BrickStatusObserver> observers = new CopyOnWriteArrayList<>();
    private volatile ModelObjects.BrickInfo brickInfo;

    public void register() {
        try {
            Logger.d(TAG, "Brick status observer attempt to register in IO SDK");
            doRegister();
            Logger.d(TAG, "Brick status observer succeeded to register in IO SDK");
        } catch (Throwable throwable) {
            Logger.e(TAG, "Brick status observer fails to register in IO SDK", throwable);
        }
    }

    @MainThread
    public void addObserver(BrickStatusObserver observer) {
        observers.add(observer);
        if (brickInfo != null) {
            observer.onBrickStatusChanged(brickInfo);
        }
    }

    @MainThread
    public void removeObserver(BrickStatusObserver observer) {
        observers.remove(observer);
    }

    public void onBrickStatusChanged(byte[] packedBrickInfo) {
        try {
            brickInfo = ModelObjects.BrickInfo.parseFrom(packedBrickInfo);
        } catch (InvalidProtocolBufferException e) {
            Logger.e(TAG, "Invalid protocol buffer: " + e.getMessage(), e);
            return;
        }

        mainHandler.post(() -> {
            Logger.i(TAG, "Got brick info from JNI: " + brickInfo);
            for (BrickStatusObserver observer : observers) {
                observer.onBrickStatusChanged(brickInfo);
            }
        });
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();
    private native void doReset();
}
