package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import com.yandex.launcher.logger.Logger;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.WorkerThread;
import ru.yandex.io.sdk.ConfigObserver;

/**
 * Observer bridge to native syncd.
 */
public class JniConfigObserver {
    private static final String TAG = "JniConfigObserver";

    private final Handler mainHandler = new Handler(Looper.getMainLooper());

    // Keep observers in same map, since same keys are not expected for different config types
    private final HashMap<String, HashSet<ConfigObserver>> observersMap = new HashMap<>();
    private final Object observersMapLock = new Object();

    /**
     * Simple caches of latest config values to provide to additional observers right away.
     * Don't expect this to grow very big.
     * Access only from main thread, so don't require additional sync
     */
    private final ConcurrentHashMap<String, String> configCache = new ConcurrentHashMap<>();

    @WorkerThread
    public void subscribeToConfig(
            @NonNull ConfigObserver configObserver,
            @NonNull String[] configNames) {
        for (String configName : configNames) {
            subscribeToConfigLocked(configObserver, configName);
        }
    }

    @WorkerThread
    public void unsubscribeFromConfig(
            @NonNull ConfigObserver configObserver) {
        unsubscribeFromConfigLocked(configObserver);
    }

    @SuppressWarnings("unused")
    public void onConfig(@NonNull String configName, @NonNull String configJsonValue) {
        Logger.d(TAG, "Config value received for %s", configName);
        runOnMainThread(() -> {
            dispatchToObservers(configName, configJsonValue);
            configCache.put(configName, configJsonValue);
        });
    }

    @MainThread
    private void dispatchToObservers(@NonNull String configName, @NonNull String configJsonValue) {
        ConfigObserver[] observers = null;

        synchronized (observersMapLock) {
            if (observersMap.containsKey(configName)) {
                observers = Objects.requireNonNull(observersMap.get(configName)).toArray(new ConfigObserver[0]);
            }
        }

        if (observers != null) {
            for (ConfigObserver observer : observers) {
                observer.onConfig(configName, configJsonValue);
            }
        }
    }

    public void register() {
        safeCallNative("register config observer", this::doRegister);
    }

    private void subscribeToConfigLocked(@NonNull ConfigObserver configObserver,
            @NonNull String configName) {
        boolean firstAddedObserver;

        synchronized (observersMapLock) {
            HashSet<ConfigObserver> observers = observersMap.get(configName);

            if (observers == null) {
                observers = new HashSet<>();
            }

            firstAddedObserver = observers.add(configObserver) && observers.size() == 1;

            observersMap.put(configName, observers);

            safeCallNative("subscribe to system config",
                    () -> doSubscribeSystem(configName));
        }

        if (!firstAddedObserver) {
            String configValue = configCache.get(configName);

            if (configValue != null) {
                Logger.d(TAG, "Dispatch %s value to new observer", configName);
                configObserver.onConfig(configName, configValue);
            }
        }
    }

    private void unsubscribeFromConfigLocked(@NonNull ConfigObserver configObserver) {
        synchronized (observersMapLock) {
            for (Map.Entry<String, HashSet<ConfigObserver>> entry : observersMap.entrySet()) {
                HashSet<ConfigObserver> observers = entry.getValue();

                if (observers.remove(configObserver) && observers.size() == 0) {
                    String key = entry.getKey();

                    safeCallNative("unsubscribe from system config",
                            () -> doUnsubscribeSystem(key));

                    // Don't keep cache for unsubscribed configs
                    configCache.remove(key);
                }
            }
        }
    }

    private void safeCallNative(@NonNull String logName, @NonNull Runnable call) {
        try {
            Logger.d(TAG, "Attempt to %s in IO SDK", logName);
            call.run();
            Logger.d(TAG, "Succeeded to %s in IO SDK", logName);
        } catch (Throwable throwable) {
            Logger.e(TAG, "Fails to " + logName + " in IO SDK", throwable);
        }
    }

    private void runOnMainThread(@NonNull Runnable r) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            r.run();
        } else {
            mainHandler.post(r);
        }
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();

    private native void doReset();

    private native void doSubscribeSystem(@NonNull String configName);

    private native void doUnsubscribeSystem(@NonNull String configName);
}
