package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.launcher.logger.Logger;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.io.sdk.DeviceGroupStateObserver;
import ru.yandex.quasar.protobuf.ModelObjects;

public class JniDeviceGroupStateObserver {
    private static final String TAG = "JniDeviceGroupStateObserver";

    private final Handler mainHandler = new Handler(Looper.getMainLooper());

    @Nullable
    private ModelObjects.DeviceGroupState currentGroupState = null;

    @NonNull
    private final List<DeviceGroupStateObserver> observers = new ArrayList<>();

    @MainThread
    public void addObserver(@NonNull DeviceGroupStateObserver observer) {
        observers.add(observer);
        if (currentGroupState != null) {
            observer.onDeviceGroupState(currentGroupState);
        }
    }

    @MainThread
    public void removeObserver(@NonNull DeviceGroupStateObserver observer) {
        observers.remove(observer);
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public void onDeviceGroupState(byte[] packedDeviceGroupState) {
        ModelObjects.DeviceGroupState unpackedDeviceGroupState;
        try {
            unpackedDeviceGroupState = ModelObjects.DeviceGroupState.parseFrom(packedDeviceGroupState);
        } catch (InvalidProtocolBufferException e) {
            Logger.e(TAG, "Invalid protocol buffer", e);
            return;
        }
        Logger.d(TAG, "Got device group state from JNI");
        mainHandler.post(() -> {
            currentGroupState = unpackedDeviceGroupState;
            for (DeviceGroupStateObserver observer : observers) {
                observer.onDeviceGroupState(currentGroupState);
            }
        });
    }

    public void register() {
        try {
            Logger.d(TAG, "Device group state observer attempt to register in IO SDK");
            doRegister();
            Logger.d(TAG, "Device group state observe succeeded to register in IO SDK");
        } catch (Throwable throwable) {
            Logger.e(TAG, "Device group state observer fails to register in IO SDK", throwable);
        }
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();
    private native void doReset();
}
