package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.launcher.logger.Logger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import ru.yandex.io.sdk.GlagolDiscoveryObserver;
import ru.yandex.quasar.protobuf.ModelObjects;

public class JniGlagolDiscoveryObserver {
    private static final String TAG = "JniGlagolDiscoveryObserver";
    private final Handler mainHandler = new Handler(Looper.getMainLooper());

    @NonNull
    private final List<GlagolDiscoveryObserver> observers = new ArrayList<>();

    @MainThread
    public void addObserver(@NonNull GlagolDiscoveryObserver observer) {
        observers.add(observer);
    }

    @MainThread
    public void removeObserver(@NonNull GlagolDiscoveryObserver observer) {
        observers.remove(observer);
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public void onDiscoveryResult(byte[] packedGlagolDiscoveryResult) {
        ModelObjects.GlagolDiscoveryResult unpackedResult;
        try {
            unpackedResult = ModelObjects.GlagolDiscoveryResult.parseFrom(packedGlagolDiscoveryResult);
        } catch (InvalidProtocolBufferException e) {
            Logger.e(TAG, "Invalid protocol buffer: " + e.getMessage(), e);
            return;
        }
        Logger.d(TAG, "Got SDK state from JNI %s", unpackedResult);
        mainHandler.post(() -> {
            for (GlagolDiscoveryObserver observer : observers) {
                observer.onDiscoveryResult(unpackedResult);
            }
        });
    }

    public void register() {
        try {
            Logger.d(TAG, "Glagol discovery observer attempt to register in IO SDK");
            doRegister();
            Logger.d(TAG, "Glagol discovery observer succeeded to register in IO SDK");
        } catch (Throwable throwable) {
            Logger.e(TAG, "Glagol discovery observer fails to register in IO SDK", throwable);
        }
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();
    private native void doReset();
}
