package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.launcher.logger.Logger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import ru.yandex.io.sdk.MusicStateObserver;
import ru.yandex.quasar.protobuf.ModelObjects;

public class JniMusicStateObserver {
    private static final String TAG = "JniMusicStateObserver";

    private final Handler mainHandler = new Handler(Looper.getMainLooper());

    @NonNull
    private final List<MusicStateObserver> observers = new ArrayList<>();

    @MainThread
    public void addObserver(@NonNull MusicStateObserver observer) {
        observers.add(observer);
    }

    @MainThread
    public void removeObserver(@NonNull MusicStateObserver observer) {
        observers.remove(observer);
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public void onMusicStateChanged(byte[] packedMusicState) {
        ModelObjects.AppState.MusicState unpackedMusicState;
        try {
            unpackedMusicState = ModelObjects.AppState.MusicState.parseFrom(packedMusicState);
        } catch (InvalidProtocolBufferException e) {
            Logger.e(TAG, "Invalid protocol buffer: " + e.getMessage(), e);
            return;
        }
        Logger.d(TAG, String.format("Got music state from JNI: reqId=%s", unpackedMusicState.getVinsRequestId()));
        mainHandler.post(() -> {
            for (MusicStateObserver observer : observers) {
                observer.onMusicStateChanged(unpackedMusicState);
            }
        });
    }

    public void register() {
        try {
            Logger.d(TAG, "Music state observer attempt to register in IO SDK");
            doRegister();
            Logger.d(TAG, "Music state observer succeeded to register in IO SDK");
        } catch (Throwable throwable) {
            Logger.e(TAG, "Music state observer fails to register in IO SDK", throwable);
        }
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();
    private native void doReset();
}
