package ru.yandex.io.sdk.jni;

import android.os.Handler;
import android.os.Looper;

import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.launcher.logger.Logger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import ru.yandex.io.sdk.SDKStateObserver;
import ru.yandex.quasar.protobuf.ModelObjects;
import ru.yandex.quasar.protobuf.YandexIO;

public class JniSDKStateObserver {
    private static final String TAG = "JniSDKStateObserver";

    private final Handler mainHandler = new Handler(Looper.getMainLooper());

    @NonNull
    private final List<SDKStateObserver> observers = new ArrayList<>();

    @MainThread
    public void addObserver(@NonNull SDKStateObserver observer) {
        observers.add(observer);
    }

    @MainThread
    public void removeObserver(@NonNull SDKStateObserver observer) {
        observers.remove(observer);
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public void onSDKStateChanged(byte[] packedSDKState) {
        YandexIO.IOEvent.SDKState unpackedState;
        try {
            unpackedState = YandexIO.IOEvent.SDKState.parseFrom(packedSDKState);
        } catch (InvalidProtocolBufferException e) {
            Logger.e(TAG, "Invalid protocol buffer: " + e.getMessage(), e);
            return;
        }
        Logger.d(TAG, "Got SDK state from JNI %s", unpackedState.getAliceState());
        mainHandler.post(() -> {
            for (SDKStateObserver observer : observers) {
                observer.onSDKStateChanged(unpackedState);
            }
        });
    }

    public void register() {
        try {
            Logger.d(TAG, "SDK state observer attempt to register in IO SDK");
            doRegister();
            Logger.d(TAG, "SDK state observer succeeded to register in IO SDK");
        } catch (Throwable throwable) {
            Logger.e(TAG, "SDK state observer fails to register in IO SDK", throwable);
        }
    }

    public void reset() {
        doReset();
    }

    private native void doRegister();
    private native void doReset();
}
