package ru.yandex.io.sdk.jni;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.List;
import java.util.UUID;

import ru.yandex.alice.megamind.protos.common.DeviceStateProto;
import ru.yandex.alice.megamind.protos.speechkit.SpeechkitRequestProto;
import ru.yandex.io.sdk.results.CallSdkResult;
import ru.yandex.io.sdk.results.RegistrationResult;
import ru.yandex.quasar.protobuf.DeviceSdkProto;
import ru.yandex.quasar.protobuf.ModelObjects;
import ru.yandex.quasar.protobuf.YandexIO;
import ru.yandex.quasar.protobuf.YandexIO.TvPolicyInfo;
import ru.yandex.alice.library.client.protos.ClientInfoProto;

public final class YandexIOSDK {
    public native void toggleConversation(byte[] eventSource);
    public native void startConversation(byte[] eventSource);
    public native void stopConversation();
    public native void stopAlarm();
    public native void finishConversationVoiceInput();
    public native void togglePlayPause();
    public native void nextTrack();
    public native void prevTrack();
    public native void pause();
    public native void like();
    public native void dislike();
    public native void revokeUserAccountInfo();
    public native void sendTextQuery(String query, @Nullable JniAliceRequestCallback callback, byte[] eventSource);

    /**
     * @param   request     string with JSON for request
     * @param   enqueued    if to enqueue the request (if false will interrupt current requests)
     * @param   parallel    if to start request ingoring current alice requests
     * @param   callback    alice request completion callback
     * @param   eventSource TEventSource protobuf
     */
    public native void sendUnstructuredServerRequest(String request, boolean enqueued, boolean parallel, @Nullable JniAliceRequestCallback callback, byte[] eventSource);

    public void authenticate(@NonNull String xcode) {
        doNativeAuthenticate(xcode);
    }

    public void playSoundFile(@NonNull String fileName, @Nullable ModelObjects.AudioChannel channel, boolean playLooped) {
        int channelInt = -1;
        if (channel != null) {
            channelInt = channel.getNumber();
        }
        doNativePlaySoundFile(fileName, channelInt, playLooped);
    }

    public void stopSoundFile(@NonNull String fileName) {
        doNativeStopSoundFile(fileName);
    }

    public void sendAudioRewind(double amountMs, @NonNull String rewindType) {
        doNativeSendAudioRewind(amountMs, rewindType);
    }

    public void sendActiveActionSemanticFrame(@Nullable String activeAction) {
        doNativeSendActiveActionSemanticFrame(activeAction);
    }

    public void sendActiveActions(@NonNull DeviceStateProto.TDeviceState.TActiveActions activeActions) {
        doNativeSendActiveActions(activeActions.toByteArray());
    }

    public void sendTvPolicyInfo(@NonNull TvPolicyInfo tvPolicyInfo) {
        doNativeSendTvPolicyInfo(tvPolicyInfo.toByteArray());
    }

    public final void provideUserAccountInfo(@NonNull String oauthToken, @NonNull String uid) {
        checkTokenUid(oauthToken, uid);
        doNativeProvideUserAccountInfo(oauthToken, uid);
    }

    public final void registerDeviceInBackend(@NonNull String oauthToken, @NonNull String uid) {
        checkTokenUid(oauthToken, uid);
        doNativeRegisterDeviceInBackend(oauthToken, uid);
    }

    public final CallSdkResult<RegistrationResult> registerDeviceInBackendSync(@NonNull String oauthToken, @NonNull String uid) {
        checkTokenUid(oauthToken, uid);
        int[] outResponseCode = new int[1];
        String[] outErrorMessage = new String[1];
        boolean isSuccess = doNativeRegisterDeviceInBackendSync(oauthToken, uid, outResponseCode, outErrorMessage);
        return new CallSdkResult<>(isSuccess, new RegistrationResult(outResponseCode[0], outErrorMessage[0]));
    }

    public final void sendIsScreenActive(boolean isScreenActive) {
        doNativeSendIsScreenActive(isScreenActive);
    }

    public void provideState(@NonNull DeviceSdkProto.TDeviceStatePart statePart) {
        doNativeProvideState(statePart.toByteArray());
    }

    public void provideMediaDeviceIdentifier(@NonNull ClientInfoProto.TClientInfoProto.TMediaDeviceIdentifier identifier) {
        doNativeProvideMediaDeviceIdentifier(identifier.toByteArray());
    }

    public void sendWifiList(@NonNull List<ModelObjects.WifiInfo> wifiList) {
        ModelObjects.WifiList wifiListProto = ModelObjects.WifiList.newBuilder().addAllHotspots(wifiList).build();
        doNativeSendWifiList(wifiListProto.toByteArray());
    }

    public void sendTimezone(@NonNull String timezone, int offsetSec) {
        doNativeSendTimezone(timezone, offsetSec);
    }

    public void sendLocation(double latitude, double longitude, @Nullable Double accuracy) {
        doNativeSendLocation(latitude, longitude, accuracy == null ? -1.0 : accuracy);
    }

    public void sendIsMicrophoneMuted(boolean muted) {
        doNativeSendIsMicrophoneMuted(muted);
    }

    public final void executeDirectiveSequence(@NonNull List<YandexIO.DirectiveEntity> directives) {
        YandexIO.DirectiveEntityList directiveEntityListProto
                = YandexIO.DirectiveEntityList.newBuilder().addAllDirectives(directives).build();
        doNativeExecuteDirectiveSequence(directiveEntityListProto.toByteArray());
    }

    private void checkTokenUid(@Nullable String oauthToken, @Nullable String uid) {
        if (oauthToken == null) {
            throw new NullPointerException("oauthToken");
        }
        if (uid == null) {
            throw new NullPointerException("uid");
        }
    }

    private native void doNativeAuthenticate(String xcode);
    private native void doNativePlaySoundFile(String fileName, int channel, boolean playLooped);
    private native void doNativeStopSoundFile(String fileName);
    private native void doNativeProvideUserAccountInfo(String oauthToken, String uid);
    private native void doNativeRegisterDeviceInBackend(String oauthToken, String uid);
    private native boolean doNativeRegisterDeviceInBackendSync(String oauthToken, String uid, int[] outResponseCode, String[] outErrorMessage);
    private native void doNativeSendTvPolicyInfo(byte[] tvPolicyInfoSerialized);
    private native void doNativeSendIsScreenActive(boolean isScreenActive);
    private native void doNativeProvideState(byte[] statePart);
    private native void doNativeProvideMediaDeviceIdentifier(byte[] identifier);
    private native void doNativeSendWifiList(byte[] wifiList);
    private native void doNativeSendTimezone(String timezone, int offsetSec);
    private native void doNativeSendLocation(double latitude, double longitude, double accuracy);
    private native void doNativeSendActiveActionSemanticFrame(String activeAction);
    private native void doNativeSendActiveActions(byte[] activeActionsSerialized);
    private native void doNativeSendAudioRewind(double amount, String rewindType);
    private native void doNativeExecuteDirectiveSequence(byte[] directives);
    private native void doNativeSendIsMicrophoneMuted(boolean muted);

    // The methods below are deprecated. Specify eventSource explicitly.
    public void toggleConversation() {
        SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource eventSource = SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.newBuilder()
                .setEvent(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EEvent.Click)
                .setSource(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.ESource.Hardware)
                .setType(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EType.Voice)
                .setId(UUID.randomUUID().toString())
                .build();
        toggleConversation(eventSource.toByteArray());
    }

    public void startConversation() {
        SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource eventSource = SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.newBuilder()
                .setEvent(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EEvent.Click)
                .setSource(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.ESource.Hardware)
                .setType(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EType.Voice)
                .setId(UUID.randomUUID().toString())
                .build();
        startConversation(eventSource.toByteArray());
    }

    public void sendTextQuery(String query, @Nullable JniAliceRequestCallback callback) {
        SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource eventSource = SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.newBuilder()
                .setEvent(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EEvent.Directive)
                .setSource(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.ESource.Software)
                .setType(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EType.Text)
                .setId(UUID.randomUUID().toString())
                .build();
        sendTextQuery(query, callback, eventSource.toByteArray());
    }

    public void sendUnstructuredServerRequest(String request, boolean enqueued, boolean parallel, @Nullable JniAliceRequestCallback callback) {
        SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource eventSource = SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.newBuilder()
                .setEvent(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EEvent.Directive)
                .setSource(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.ESource.Software)
                .setType(SpeechkitRequestProto.TSpeechKitRequestProto.TEventSource.EType.Text)
                .setId(UUID.randomUUID().toString())
                .build();
        sendUnstructuredServerRequest(request, enqueued, parallel, callback, eventSource.toByteArray());
    }
}
