package ru.yandex.io.sdk.jni;

import androidx.annotation.NonNull;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.OneofDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message;
import com.yandex.launcher.logger.Logger;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import ru.yandex.alice.protos.endpoint.capabilities.AllCapabilitiesProto;
import ru.yandex.alice.protos.endpoint.EndpointProto.TEndpoint;
import ru.yandex.alice.protos.endpoint.EndpointProto.TEndpoint.TMeta;
import ru.yandex.io.sdk.capability.Capability;
import ru.yandex.io.sdk.capability.IoEndpoint;

public final class YandexIoEndpoint extends IoEndpoint implements Capability.Listener {
    private static final String TAG = "YandexIoEndpoint";
    private final String deviceId;
    private final TEndpoint.EEndpointType type;
    private final List<Capability> capabilities = new CopyOnWriteArrayList<>();
    private final List<Listener> listeners = new CopyOnWriteArrayList<>();

    public YandexIoEndpoint(@NonNull String deviceId, @NonNull TEndpoint.EEndpointType type) {
        this.deviceId = deviceId;
        this.type = type;

        listeners.add(new Listener() {
            @Override
            public void onCapabilityAdded(@NonNull Capability capability) {
                YandexIoEndpoint.this.nativeAddCapability(capability);
            }

            @Override
            public void onCapabilityRemoved(@NonNull Capability capability) {
                YandexIoEndpoint.this.nativeRemoveCapability(capability.getId());
            }
        });
    }

    @NonNull
    public String getId() {
        return deviceId;
    }

    @NonNull
    public TEndpoint getState() {
        TEndpoint.Builder stateBuilder = TEndpoint.newBuilder()
                .setId(getId())
                .setMeta(TMeta.newBuilder().setType(type));

        for (Capability capability : capabilities) {
            Message capabilityState = getCapabilityFromHolder(capability.getState());
            stateBuilder.addCapabilities(Any.pack(capabilityState));
        }
        return stateBuilder.build();
    }

    public byte[] getEndpointState() {
        return getState().toByteArray();
    }

    @Override
    public void addCapability(@NonNull Capability capability) {
        Logger.d(TAG, "Adding capability with id " + capability.getId());

        capability.addListener(this);

        capabilities.add(capability);
        for (Listener listener : listeners) {
            listener.onCapabilityAdded(capability);
        }
    }

    @Override
    public void removeCapability(@NonNull Capability capability) {
        Logger.d(TAG, "Removing capability with id " + capability.getId());

        capability.removeListener(this);

        capabilities.remove(capability);
        for (Listener listener : listeners) {
            listener.onCapabilityRemoved(capability);
        }
    }

    @Override
    @NonNull
    public List<Capability> getCapabilities() {
        return capabilities;
    }

    @Override
    public void addListener(@NonNull Listener listener) {
        listeners.add(listener);
    }

    @Override
    public void removeListener(@NonNull Listener listener) {
        listeners.add(listener);
    }

    @Override
    public void onCapabilityStateChanged(@NonNull Capability capability, @NonNull AllCapabilitiesProto.TCapabilityHolder state) {
        nativeOnCapabilityStateChanged(capability.getId(), state.toByteArray());
    }

    private Message getCapabilityFromHolder(AllCapabilitiesProto.TCapabilityHolder state) {
        OneofDescriptor oneofDescriptor = state.getDescriptorForType().getOneofs().get(0);
        FieldDescriptor fieldDescriptor = state.getOneofFieldDescriptor(oneofDescriptor);
        return (Message) state.getField(fieldDescriptor);
    }

    private native void nativeAddCapability(Capability capability);

    private native void nativeRemoveCapability(String capabilityId);

    private native void nativeOnCapabilityStateChanged(String capabilityId, byte[] capabilityState);
}
