#include "speechkit_audio_player.h"

#include <yandex_io/android_sdk/libs/cpp/interfaces/audio_sink.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/base/named_callback_queue.h>

#include <set>

namespace YandexIO {
    using WeakCmp = std::owner_less<std::weak_ptr<SpeechKit::AudioPlayer::AudioPlayerListener>>;

    class SpeechkitAudioPlayer: public SpeechKit::AudioPlayer, public std::enable_shared_from_this<SpeechkitAudioPlayer> {
    public:
        explicit SpeechkitAudioPlayer(std::shared_ptr<IAudioSink> sink)
            : SpeechKit::AudioPlayer()
            , sink_(std::move(sink))
            , commandsQueue_("speechkit_audio_player")
        {
        }

        ~SpeechkitAudioPlayer() override {
            commandsQueue_.clear();
        }

        void play() override {
            commandsQueue_.add(std::bind(&SpeechkitAudioPlayer::doPlay, this));
        }

        void pause() override {
            commandsQueue_.add(std::bind(&SpeechkitAudioPlayer::doPause, this));
        }

        void cancel() override {
            commandsQueue_.clear();
            commandsQueue_.add(std::bind(&SpeechkitAudioPlayer::doCancel, this));
        }

        void playData(SpeechKit::SoundBuffer::SharedPtr buffer) override {
            const auto& info = buffer->getInfo();
            const auto samplesPerChunk = info.getSampleRate() / 10;
            const auto bytesPerChunk = samplesPerChunk * info.getChannelCount() * info.getSampleSize();
            for (std::size_t pos = 0, limit = buffer->getData().size(); pos < limit; pos += bytesPerChunk) {
                commandsQueue_.add(std::bind(&SpeechkitAudioPlayer::doPlayData, this, buffer, pos, bytesPerChunk));
            }
        }

        void setDataEnd() override {
            commandsQueue_.add(std::bind(&SpeechkitAudioPlayer::doStop, this));
        }

        void setVolume(float gain) override {
            YIO_LOG_TRACE("setVolume" << gain);
        }

        float getVolume() const override {
            YIO_LOG_TRACE("getVolume");
            return {};
        }

        void subscribe(AudioPlayerListener::WeakPtr listener) override {
            YIO_LOG_TRACE("subscribe");
            std::scoped_lock lock(listeners_guard_);
            listeners_.insert(std::move(listener));
        }

        void unsubscribe(AudioPlayerListener::WeakPtr listener) override {
            YIO_LOG_TRACE("unsubscribe");
            std::scoped_lock lock(listeners_guard_);
            listeners_.erase(listener);
        }

    private:
        void doPlay() {
            if (switchState(State::PAUSED, State::PLAYING)) {
                notifyListeners(&AudioPlayerListener::onPlayingResumed);
                sink_->resume();
            }
        }

        void doPause() {
            if (switchState(State::PLAYING, State::PAUSED)) {
                notifyListeners(&AudioPlayerListener::onPlayingPaused);
                sink_->pause();
            }
        }

        void doCancel() {
            if (switchState(State::PLAYING, State::STOPPED) || switchState(State::PAUSED, State::STOPPED)) {
                sink_->cancel();
            }
        }

        void doPlayData(SpeechKit::SoundBuffer::SharedPtr buffer, std::size_t offset, std::size_t size) {
            if (switchState(State::STOPPED, State::PLAYING)) {
                notifyListeners(&AudioPlayerListener::onPlayingBegin);
                const auto& info = buffer->getInfo();
                sink_->start(info.getChannelCount(), info.getSampleRate(), info.getSampleSize());
            }
            sink_->pushData(std::span<const std::uint8_t>{buffer->getData()}.subspan(offset, size));
        }

        void doStop() {
            if (switchState(State::PLAYING, State::STOPPED) || switchState(State::PAUSED, State::STOPPED)) {
                notifyListeners(&AudioPlayerListener::onPlayingDone);
                sink_->finish();
            }
        }

        template <class Method>
        void notifyListeners(Method method) {
            std::scoped_lock lock(listeners_guard_);
            for (const auto& entry : listeners_) {
                if (auto slistener = entry.lock()) {
                    (slistener.get()->*method)(shared_from_this());
                }
            }
        }

        enum class State: int {
            STOPPED,
            PLAYING,
            PAUSED,
        };

        bool switchState(State from, State to) {
            return state_.compare_exchange_strong(from, to);
        }

    private:
        const std::shared_ptr<IAudioSink> sink_;
        std::mutex listeners_guard_;
        std::set<AudioPlayerListener::WeakPtr, WeakCmp> listeners_;
        quasar::NamedCallbackQueue commandsQueue_;
        std::atomic<State> state_{State::STOPPED};
    };

    SpeechKit::AudioPlayer::SharedPtr createSpeechkitAudioPlayer(std::shared_ptr<IAudioSink> sink) {
        return std::make_shared<SpeechkitAudioPlayer>(std::move(sink));
    }
} // namespace YandexIO
