#include "voice_sink.h"

#include <string>

#include <yandex_io/android_sdk/libs/cpp/interfaces/audio_sink.h>
#include <yandex_io/modules/audio_input/channels_splitter/channels_splitter.h>
#include <yandex_io/sdk/audio_source/i_audio_source.h>
#include <util/generic/yexception.h>

namespace YandexIO {
    class VoiceSink: public IAudioSink {
    public:
        explicit VoiceSink(std::shared_ptr<IAudioSource> delegate)
            : delegate_(std::move(delegate))
        {
        }

        void start(int channels, int sampleRate, int sampleSize) override {
            Y_ENSURE(sampleSize == sizeof(ChannelData::SampleInt), "Unsupported sample size");
            channels_ = channels;
            samplerate_ = sampleRate;
        }
        void pause() override {
        }
        void resume() override {
        }
        void cancel() override {
            channels_ = 0;
            samplerate_ = 0;
        }
        void finish() override {
        }
        void pushData(std::span<const std::uint8_t> data) override {
            Y_ENSURE(channels_ != 0 && samplerate_ != 0, "start() is not called");
            auto output = convert({reinterpret_cast<const ChannelData::SampleInt*>(data.data()), data.size() / sizeof(ChannelData::SampleInt)});
            delegate_->pushData(std::move(output));
        }

    private:
        ChannelsData convert(std::span<const ChannelData::SampleInt> interleaved) {
            if (channels_ == 1) {
                return convertMono(interleaved);
            } else {
                ChannelsData result;
                for (int index = 0; index < channels_; ++index) {
                    const std::array<int, 1> inputChannelsIndices = {index};
                    auto channel = allocateChannelData(index, interleaved.size() / channels_);
                    audio::extractChannels(interleaved, channels_, inputChannelsIndices, {channel.data});
                    result.emplace_back(std::move(channel));
                }
                return result;
            }
        }

        ChannelsData convertMono(std::span<const ChannelData::SampleInt> interleaved) {
            auto channel = allocateChannelData(0, interleaved.size());
            std::memcpy(channel.data.data(), interleaved.data(), interleaved.size_bytes());
            return {std::move(channel)};
        }

        ChannelData allocateChannelData(int index, std::size_t samples) const {
            ChannelData result;
            result.name = std::to_string(index);
            result.type = ChannelData::Type::VQE;
            result.sampleRate = samplerate_;
            result.isForRecognition = true;
            result.data.resize(samples);
            return result;
        }

    private:
        const std::shared_ptr<IAudioSource> delegate_;
        int channels_{0};
        int samplerate_{0};
    };

    std::shared_ptr<IAudioSink> createVoiceSink(std::shared_ptr<IAudioSource> source) {
        return std::make_shared<VoiceSink>(std::move(source));
    }
} // namespace YandexIO
