#include "jni_audio_sink.h"
#include "jni_exception.h"

#include <yandex_io/android_sdk/libs/cpp/interfaces/audio_sink.h>
#include <library/cpp/jni/jni.h>

namespace YandexIO {
    class JniAudioSink: public IAudioSink {
    public:
        explicit JniAudioSink(jobject delegate)
            : env_(*NJni::TJniEnv::Get())
            , delegate_(delegate)
        {
            const auto delegateClass = env_.FindClass("com/yandex/io/AudioSink");
            start_ = env_.GetMethodID(delegateClass.Get(), "start", "(III)V", false);
            pause_ = env_.GetMethodID(delegateClass.Get(), "pause", "()V", false);
            resume_ = env_.GetMethodID(delegateClass.Get(), "resume", "()V", false);
            cancel_ = env_.GetMethodID(delegateClass.Get(), "cancel", "()V", false);
            finish_ = env_.GetMethodID(delegateClass.Get(), "finish", "()V", false);
            pushData_ = env_.GetMethodID(delegateClass.Get(), "pushData", "(Ljava/nio/ByteBuffer;)V", false);
        }

        void start(int channels, int sampleRate, int sampleSize) override {
            env_.CallVoidMethod(delegate_.Get(), start_, channels, sampleRate, sampleSize);
        }

        void pause() override {
            env_.CallVoidMethod(delegate_.Get(), pause_);
        }

        void resume() override {
            env_.CallVoidMethod(delegate_.Get(), resume_);
        }

        void cancel() override {
            env_.CallVoidMethod(delegate_.Get(), cancel_);
        }

        void finish() override {
            env_.CallVoidMethod(delegate_.Get(), finish_);
        }

        void pushData(std::span<const std::uint8_t> data) override {
            env_.CallVoidMethod(delegate_.Get(), pushData_, ToByteBuffer(data));
        }

    private:
        jobject ToByteBuffer(std::span<const std::uint8_t> data) const {
            return env_.GetJniEnv()->NewDirectByteBuffer(const_cast<void*>(static_cast<const void*>(data.data())), data.size());
        }

    private:
        NJni::TJniEnv& env_;
        const NJni::TGlobalRef delegate_;
        jmethodID start_;
        jmethodID pause_;
        jmethodID resume_;
        jmethodID cancel_;
        jmethodID finish_;
        jmethodID pushData_;
    };

    std::shared_ptr<IAudioSink> createJavaAudioSink(jobject sink) {
        return std::make_shared<JniAudioSink>(sink);
    }
}
